(ns quantus.util
  (:require [quantus.core :as q])
  (:import [quantus.core Quantity]))

(defn quantity?
  [q]
  (= Quantity (class q)))

(defn quantity-wrapper
  "Wraps a function so that it can operate on Quantities.  Returns a
  function that, given multiple inputs, one or more of which may be
  quantities, evaluates the function `f` on the values of the input
  quantities, and converts the output to a quantity whose unit matches
  the first input.  Also verifies that all inputs use the same unit
  type (for those inputs that are quantities).

  Options can be passed as a map for the second input.

  - `:out` can be either `:quantity` (default), or `:collection`.  If
  collection, output will be iterated over and converted to a
  quantity.

  Tip: if you want the unit of one of the input quantities to be ignored, unwrap it yourself."
  ([f] (quantity-wrapper f {}))
  ([f {output-type :out
       :or {output-type :quantity}}]
   (assert (#{:quantity :collection} output-type))
   (fn [& args]
     (let [unit (q/unit-type (first args))] ;; TODO: what if the first input is not a quantity?  What if none are?
       (doseq [arg (rest args)] ;; TODO: extend to collection inputs
         (when (quantity? arg)
           (q/assert-unit-type-match (first args) arg)))
       (let [out (apply f (map #(if (quantity? %) (q/value %) %) args))]
         (case output-type
           :quantity (q/Quantity. out unit)
           :collection (into (empty out)
                             (map #(q/Quantity. % unit))
                             out)))))))
