;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns servo-rn.connection
  (:require [servo-rn.reserved :as r]
            [signum.signal :refer [signal alter!]]
            [react-native-sqlite-2 :default sqlite]
            [tempus.core :as t :refer [DateTime]]
            [tempus.transit :as tt]
            [cognitect.transit :as transit]
            [cljs.pprint :refer [pprint]]
            [squel :as squel]
            [utilis.map :refer [map-keys map-vals]]
            [utilis.inflections :as i]
            [utilis.js :as j]
            [integrant.core :as ig]
            [clojure.string :as st]))

(declare connect disconnect ->sql-expr dispose
         create-tables create-indices
         sql-result->clj throw-and-warn
         with-lookups ->sql-name ->sql-value
         mutate? query->table refresh count? op?
         await-table encode decode)

(defmethod ig/init-key :servo-rn/connection
  [_ {:keys [db-name
             db-version
             db-display-name
             db-max-size-bytes
             on-ready on-error
             tables indices]
      :as options}]
  (try (connect options)
       (catch :default e
         (js/console.warn e)
         (throw e))))

(defmethod ig/halt-key! :servo-rn/connection
  [_ connection]
  (disconnect connection))

(defn connect
  [{:keys [db-name
           db-version
           db-display-name
           db-max-size-bytes
           on-ready on-error
           tables indices]
    :or {db-version "1.0"
         db-display-name "Servo RN Database"}
    :as config}]
  (let [{:keys [tables] :as config} (update config :tables with-lookups)
        ready (atom false)
        error (atom nil)
        conn-atom (atom nil)
        db (j/call sqlite :openDatabase
                   db-name
                   db-version
                   db-display-name
                   db-max-size-bytes
                   #(let [ready-conn (assoc @conn-atom :ready (atom true))]
                      (-> ready-conn
                          (create-tables tables)
                          (j/call :then (fn [] (create-indices ready-conn indices)))
                          (j/call :then (fn []
                                          (reset! ready true)
                                          (when on-ready
                                            (on-ready))))))
                   #(do (reset! error %)
                        (when on-error
                          (on-error %))))
        conn (merge config
                    {:db db
                     :db-version db-version
                     :db-display-name db-display-name
                     :table->callbacks (atom {})
                     :ready ready
                     :error error})]
    (reset! conn-atom conn)
    conn))

(defn disconnect
  [{:keys [db ready]}]
  (reset! ready false)
  (j/call db :close))

(defn delete
  [{:keys [db db-name]}]
  (js/Promise.
   (fn [resolve reject]
     (j/call sqlite :deleteDatabase
             db-name
             (fn [] (resolve))
             (reject)))))

(defn on-ready
  [{:keys [ready]} f]
  (if @ready
    (f)
    (let [key (str (random-uuid))]
      (add-watch ready key
                 (fn [_ _ _ ready?]
                   (when ready?
                     (f)
                     (remove-watch ready key)))))))

(defn run-sql-expr
  [{:keys [db]} sql-expr]
  (js/Promise.
   (fn [resolve reject]
     (j/call db :transaction
             (fn [tx]
               (j/call tx :executeSql
                       sql-expr #js []
                       (fn [_tx result]
                         (resolve result))
                       reject))))))

(defn run
  [{:keys [db] :as connection} query]
  (let [mutate? (mutate? query)]
    (when (not connection)
      (throw-and-warn "No connection provided" {:query query}))
    (js/Promise.
     (fn [resolve reject]
       (on-ready
        connection
        (fn []
          (try (if-let [sql-expr (->sql-expr connection query)]
                 (j/call db :transaction
                         (fn [tx]
                           (j/call tx :executeSql
                                   sql-expr #js []
                                   (fn [_tx result]
                                     (js/setTimeout
                                      (fn []
                                        (try (when mutate?
                                               (some->> query
                                                        query->table
                                                        (refresh connection)))
                                             (->> result
                                                  (sql-result->clj connection query)
                                                  resolve)
                                             (catch :default e
                                               (reject e))))))
                                   (fn [_tx error]
                                     (reject error)))))
                 (resolve nil))
               (catch :default e
                 (reject e)))))))))

(defn subscribe
  [connection query]
  (cond
    (mutate? query)
    (throw-and-warn
     "Can not subscribe to a mutating query"
     {:query query})

    query
    (let [signal (signal)
          table (query->table query)
          callback (fn []
                     (-> connection
                         (run query)
                         (j/call :then #(alter! signal (constantly %)))
                         (j/call :catch #(alter! signal (constantly %))))
                     nil)]
      (swap! (:table->callbacks connection) update
             table conj {:signal signal
                         :callback callback
                         :query query})
      (callback)
      signal)

    :else (throw-and-warn "Got nil query in servo-rn/subscribe")))

(defn dispose
  [connection signal]
  (swap! (:table->callbacks connection)
         (partial map-vals
            (partial remove (comp (partial = signal) :signal)))))

(defn ensure-table
  [connection table-name]
  (-> connection
      (run [[:table-create table-name]])
      (j/call :then #(await-table connection table-name))))


;;; Private

(defn ->sql-name
  [v]
  (when v
    (let [result (name (i/underscore v))]
      (when (re-find #"'" result)
        (throw-and-warn "SQL names can not have single quotes"
                        {:name v
                         :sql-name result}))
      (r/->sql result))))

(defn sql-name->
  [v]
  (keyword (i/hyphenate (r/sql-> v))))

(defn- sanitize-string
  [s]
  (st/replace s "'" "''"))

(defn ->sql-value
  ([v] (let [v (cond
                 (keyword? v) (name v)
                 (boolean? v) (if v 1 0)
                 (instance? DateTime v) (t/into :long v)
                 (string? v) v
                 (nil? v) v
                 (coll? v) (encode v)
                 (number? v) v
                 :else (throw-and-warn
                        "Unsupported value type"
                        {:value v}))]
         (if (string? v)
           (sanitize-string v)
           v)))
  ([type v]
   (if (= type :serialized)
     (sanitize-string (encode v))
     (->sql-value v))))

(defn sql-value->
  [type v]
  (condp = type
    :text v
    :integer v
    :number v
    :date-time (t/from :long v)
    :keyword (when (string? v)
               (keyword v))
    :serialized (decode v)
    :boolean (= 1 v)
    (throw-and-warn
     "Unrecognized servo schema type in sql-value->"
     {:type type
      :v v})))

(defn ->sql-type
  [v]
  (condp = v
    :text "TEXT"
    :integer "INTEGER"
    :number "REAL"
    :date-time "INTEGER"
    :keyword "TEXT"
    :serialized "TEXT"
    :boolean "INTEGER"
    (throw-and-warn
     "Unrecognized servo schema type in ->sql-type"
     {:type v})))

(defn ->sql-row
  [{:keys [primary-key key->type]} clj-row]
  (when (and primary-key
             (not (get clj-row primary-key)))
    (throw-and-warn
     "No primary key provided for row"
     {:row clj-row
      :primary-key primary-key}))
  (->> clj-row
       (reduce (fn [result [k v]]
                 (assoc! result
                         (->sql-name k)
                         (->sql-value (key->type k) v)))
               (transient {}))
       (persistent!)))

(defn sql-row->
  [{:keys [key->type]} _query sql-row]
  (->> (js->clj sql-row :keywordize-keys true)
       (reduce (fn [result [k v]]
                 (let [k (sql-name-> k)]
                   (assoc! result
                           k
                           (sql-value->
                            (key->type k) v))))
               (transient {}))
       persistent!))

(defn index-create
  [{:keys [tables]} _expr _result [_ table column]]
  (when (not (get tables table))
    (throw-and-warn
     "Table does not exist in index-create"
     {:table table
      :column column}))
  {:squel (str "CREATE INDEX IF NOT EXISTS '"
               (->sql-name column)
               "' ON "
               (->sql-name table)
               " ( "
               (->sql-name column)
               " );")})

(defn compile-create-table
  [{:keys [tables]} {:keys [table]}]
  (let [{:keys [schema]} (get tables table)]
    (when (not schema)
      (throw-and-warn
       "A schema must be provided when creating a table"
       {:table table
        :schema schema}))
    {:squel (str "CREATE TABLE IF NOT EXISTS '"
                 (->sql-name table)
                 "' ("
                 (->> schema
                      (map (fn [col]
                             (let [[key options type] (if (= 3 (count col))
                                                        col
                                                        [(first col) nil (last col)])
                                   {:keys [primary-key]} options]
                               (->> (concat
                                     [(str "'" (->sql-name key) "'")
                                      (->sql-type type)]
                                     (when primary-key
                                       ["PRIMARY KEY"]))
                                    (st/join " "))
                               )))
                      (st/join ", "))
                 ");")}))

(defn ensure-op
  [result query]
  (if (:squel result)
    result
    (let [{:keys [table]} result]
      (if (op? query :delete)
        (assoc result :squel
               (-> (squel/delete)
                   (j/call :from (->sql-name table))))
        (assoc result :squel
               (-> (squel/select (if (count? (:query result))
                                   #js {:autoQuoteFieldNames false}
                                   #js {:autoQuoteTableNames true
                                        :autoQuoteFieldNames false}))
                   (j/call :from (->sql-name table))))))))

(defn error!
  [expr [op & args] message]
  (throw-and-warn
   (str "Unable to compile expression: " message "")
   {:expr expr
    :op (into [op] args)}))

(defn table-create
  [conn _expr result [_op & args]]
  (->> (first args)
       (assoc result :table)
       (compile-create-table conn)))

(defn table
  [conn expr result [_op & args]]
  (if (= 1 (count expr))
    {:squel (->sql-expr conn (conj expr [:select]))}
    (assoc result
           :table (first args)
           :query expr)))

(defn select
  [_conn expr result _]
  (ensure-op result expr))

(defn where
  [expr key op value]
  (let [key (->sql-name key)
        value (->sql-value value)]
    (case op
      := (if (nil? value)
           (j/call expr :where (str key " IS NULL"))
           (j/call expr :where (str key " = ?") value))
      :> (j/call expr :where (str key " > ?") value)
      :>= (j/call expr :where (str key " >= ?") value)
      :< (j/call expr :where (str key " < ?") value)
      :<= (j/call expr :where (str key " <= ?") value))))

(defn compile-get
  [{:keys [tables] :as conn} expr result [_ value]]
  (-> conn
      (select expr result nil)
      (update :squel
              where (get-in tables [(:table result) :primary-key])
              :=
              value)))

(defn fill-in-keys
  [key-list m]
  (->> key-list
       (reduce (fn [m k]
                 (if (contains? m k)
                   m
                   (assoc! m k nil)))
               (transient m))
       persistent!))

(defn insert
  [{:keys [tables]} expr result [_op & args :as clause]]
  (when-let [rows (not-empty
                   (cond
                     (map? (first args))
                     [(first args)]

                     (coll? (first args))
                     (first args)

                     :else
                     (error!
                      expr clause
                      "first argument to insert must be either a map or a collection of maps.")))]
    (let [{:keys [table]} result
          table-config (get tables table)
          row-keys (distinct (mapcat keys rows))]
      (assoc result :squel
             (-> (squel/insert #js {:autoQuoteTableNames true
                                    :autoQuoteFieldNames false})
                 (j/call :into (->sql-name table))
                 (j/call :setFieldsRows
                         (->> rows
                              (map (comp (partial ->sql-row table-config)
                                      (partial fill-in-keys row-keys)))
                              clj->js)))))))

(defn upsert
  [{:keys [tables] :as conn} expr result [_ & [docs]]]
  (when-let [docs (not-empty (if (map? docs) [docs] docs))]
    (let [primary-key (get-in tables [(:table result) :primary-key])]
      (when (not (every? #(contains? % primary-key) docs))
        (throw-and-warn "Every document in an upsert must have a primarey key"
                        {:docs docs}))
      (let [insert-query [(first expr)
                          [:insert docs]]
            insert-expr (->sql-expr conn insert-query)
            update-expr (-> (fn [expr key]
                              (j/call expr :set
                                      (str key " = excluded." key)))
                            (reduce (-> (squel/update #js {:autoQuoteTableNames true})
                                        (j/call :table (->sql-name (:table result))))
                                    (->> (get-in tables [(:table result) :schema])
                                         (remove (comp (partial = primary-key) first))
                                         (map (comp ->sql-name first))))
                            (j/call :toString)
                            (st/replace
                             (str "UPDATE `"
                                  (->sql-name (:table result))
                                  "` SET")
                             "UPDATE SET"))]
        {:squel (str insert-expr
                     " ON CONFLICT (`"
                     (->sql-name primary-key)
                     "`) DO "
                     update-expr)}))))

(defn order-by
  [_conn expr result [_op & args :as clause]]
  (let [result (ensure-op result expr)
        order-by (first args)
        [field direction] (if (sequential? order-by)
                            order-by
                            [order-by :asc])]
    (when (not (#{:asc :desc} direction))
      (error! expr clause "order-by direction must be either :asc or :desc"))
    (update result :squel
            j/call :order
            (->sql-name field)
            (= direction :asc))))

(defn limit
  [_conn expr result [_op & args :as clause]]
  (let [result (ensure-op result expr)
        limit (first args)]
    (when (not (pos-int? limit))
      (error! expr clause "limit argument must be a positive integer"))
    (update result :squel j/call :limit limit)))

(defn offset
  [_conn expr result [_op & args :as clause]]
  (let [result (ensure-op result expr)
        offset (first args)]
    (when (or (not (integer? offset))
              (neg? offset))
      (error! expr clause "offset argument must be an integer >= 0"))
    (update result :squel j/call :offset offset)))

(defn compile-filter
  [_conn expr result [_ matches]]
  (reduce (fn [result [key value]]
            (update result :squel where key := value))
          (ensure-op result expr)
          matches))

(defn gte
  [_conn expr result [_ key value]]
  (-> result
      (ensure-op expr)
      (update :squel where key :>= value)))

(defn gt
  [_conn expr result [_ key value]]
  (-> result
      (ensure-op expr)
      (update :squel where key :> value)))

(defn lte
  [_conn expr result [_ key value]]
  (-> result
      (ensure-op expr)
      (update :squel where key :<= value)))

(defn lt
  [_conn expr result [_ key value]]
  (-> result
      (ensure-op expr)
      (update :squel where key :< value)))

(defn compile-count
  [_conn expr result _clause]
  (-> result
      (ensure-op expr)
      (update :squel j/call :field "COUNT(*)" "count")))

(defn compile-delete
  [{:keys [tables]} expr result [_ ids]]
  (let [ids (if (string? ids) [ids] ids)
        primary-key (get-in tables [(:table result) :primary-key])
        primary-key-sql (->sql-name primary-key)]
    (when (not primary-key)
      (throw-and-warn "No primary-key for table when performing delete"
                      {:table (:table result)
                       :query expr}))
    (let [result (ensure-op result expr)]
      (if (seq ids)
        (update result :squel j/call :where
                (reduce (fn [expr id]
                          (j/call expr :or
                                  (str primary-key-sql " = ?")
                                  id))
                        (squel/expr)
                        ids))
        result))))

(defn ->sql-expr
  [conn expr]
  (let [expr (vec expr)]
    (some-> (fn [result [op & args :as clause]]
              (condp = op
                :table-create (table-create conn expr result clause)
                :index-create (index-create conn expr result clause)
                :table (table conn expr result clause)
                :select (select conn expr result clause)
                :get (compile-get conn expr result clause)
                :insert (insert conn expr result clause)
                :upsert (upsert conn expr result clause)
                :order-by (order-by conn expr result clause)
                :limit (limit conn expr result clause)
                :offset (offset conn expr result clause)
                :filter (compile-filter conn expr result clause)
                :gte (gte conn expr result clause)
                :gt (gt conn expr result clause)
                :lte (lte conn expr result clause)
                :lt (lt conn expr result clause)
                :count (compile-count conn expr result clause)
                :delete (compile-delete conn expr result clause)
                (throw-and-warn
                 "Unable to compile expression: "
                 {:expr expr
                  :op (into [op] args)})))
            (reduce {:squel nil} expr)
            :squel
            (j/call :toString))))

(defn await-table
  [{:keys [db]} table-name]
  (js/Promise.
   (fn [resolve reject]
     (let [check-expr (str "SELECT name FROM sqlite_master WHERE type='table' AND name='"
                           (->sql-name table-name)
                           "';")
           check (fn check []
                   (j/call db :transaction
                           (fn [tx]
                             (j/call tx :executeSql
                                     check-expr #js []
                                     (fn [_tx result]
                                       (if (pos? (j/get-in result [:rows :length]))
                                         (resolve true)
                                         (js/setTimeout (fn [] (check)) 100)))
                                     reject))))]
       (check)))))

(defn- create-tables
  [conn tables]
  (->> (mapv (fn [table-name]
               (ensure-table conn table-name))
             (keys tables))
       clj->js
       js/Promise.all))

(defn- create-indices
  [connection indices]
  (->> indices
       (map (fn [[table column]]
              (run connection
                [[:index-create table column]])))
       clj->js
       js/Promise.all))

(defn query->table
  [query]
  (when (= :table (ffirst query))
    (second (first query))))

(defn op?
  [query ops]
  (->> query
       (some (comp (if (coll? ops)
                  (set ops)
                  #{ops})
                first))
       boolean))

(defn mutate?
  [query]
  (op? query #{:upsert :insert :update :delete}))

(defn count?
  [query]
  (op? query #{:count}))

(defn- sql-result->clj
  [conn query result]
  (let [row-count (j/get-in result [:rows :length])
        table (query->table query)
        table (-> conn
                  (get-in [:tables table])
                  (assoc :name table))]
    (when (pos? row-count)
      (let [item #(j/call-in result [:rows :item] %)]
        (cond
          (count? query) (j/get (item 0) :count)
          (op? query :get) (sql-row-> table query (item 0))
          :else (loop [i 0
                       result (transient [])]
                  (if (< i row-count)
                    (let [row (sql-row-> table query (item i))]
                      (recur (inc i)
                             (conj! result row)))
                    (persistent! result))))))))

(defn- throw-and-warn
  ([message] (throw-and-warn message {}))
  ([message data]
   (let [message (str message ": " (pr-str data))]
     (js/console.warn message)
     (throw (ex-info message data))
     nil)))

(defn- primary-key
  [{:keys [schema]}]
  (or (->> schema
           (filter (fn [[_ & args]]
                     (when (= 2 (count args))
                       (:primary-key (first args)))))
           ffirst)
      :id))

(defn- with-lookups
  [tables]
  (map-vals
   (fn [{:keys [schema] :as table}]
     (assoc table
            :primary-key (primary-key table)
            :key->type (->> schema
                            (map (fn [[key & args]]
                                   [key (last args)]))
                            (into {}))))
   tables))

(defn- refresh
  [connection table]
  (try (doseq [{:keys [callback]} (get @(:table->callbacks connection) table)]
         (js/Promise.
          (fn [resolve _]
            (try (callback)
                 (catch :default e
                   (js/console.warn e))
                 (finally
                   (resolve))))))
       (catch js/Error e
         (js/console.warn e)
         (pprint
          {:table->callbacks-queries
           (->> @(:table->callbacks connection)
                (take 100)
                (map :query))}))))

(defn encode [x]
  (let [handlers {:handlers (:write tt/handlers)}]
    (transit/write
     (transit/writer :json handlers) x)))

(defn decode [^String x]
  (let [handlers {:handlers (:read tt/handlers)}]
    (transit/read
     (transit/reader :json handlers) x)))
