;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns servo-rn.compile
  (:require [servo-rn.util.sql :refer [double-quote sanitize-string]]
            [servo-rn.util.query :refer [mutate? count?]]
            [servo-rn.schema :as s]
            [honey.sql :as sql]
            [utilis.string :refer [format]]))

(declare table-create index-create table columns add-column
         get* get-all insert update* order-by limit
         offset filter* gte gt lte lt
         count* delete*
         throw-and-warn
         throw-on-single-quote
         concat-where)

(defn compile
  [connection query]
  (when (not (s/query-validator query))
    (throw
     (js/Error.
      (str "Schema error occurred compiling query: "
           {:query query
            :explain (s/query-explainer query)}))))
  (let [sql-map (reduce
                 (fn [sql-map [op & _ :as clause]]
                   (case op
                     :table-create (table-create sql-map clause)
                     :index-create (index-create sql-map clause)
                     :table (table sql-map clause)
                     :get (get* sql-map clause)
                     :get-all (get-all sql-map clause)
                     :insert (insert sql-map clause)
                     :update (update* sql-map clause)
                     :order-by (order-by sql-map clause)
                     :limit (limit sql-map clause)
                     :offset (offset sql-map clause)
                     :filter (filter* sql-map clause)
                     :gte (gte sql-map clause)
                     :gt (gt sql-map clause)
                     :lte (lte sql-map clause)
                     :lt (lt sql-map clause)
                     :count (count* sql-map clause)
                     :delete (delete* sql-map clause)
                     :columns (columns sql-map clause)
                     :add-column (add-column sql-map clause)))
                 {::query query
                  ::connection connection}
                 query)
        complete (fn [sql-map]
                   (assoc sql-map :sql-expr
                          (-> sql-map
                              (dissoc ::connection ::query ::table)
                              (sql/format {:quoted true
                                           :dialect :ansi}))))]
    (if (map? sql-map)
      (complete sql-map)
      (mapv complete sql-map))))

(defn fields
  [connection table]
  (remove (partial = :id)
          (get-in connection [:tables table :field-sql-names])))

(defn sql-name
  ([connection table]
   (get-in connection [:tables table :sql-name]))
  ([connection table key]
   (get-in connection [:tables table :fields key :sql-name])))

(defn sql-value
  [connection table key value]
  ((get-in connection [:tables table :fields key :value->sql]) value))

(defn sql-type
  [connection table key]
  (get-in connection [:tables table :fields key :sql-type]))

;;; Private

(defn- ensure-select
  [{::keys [connection table] :as sql-map}]
  (cond-> sql-map
    (not (:from sql-map))
    (assoc :from (sql-name connection table))

    (not (:select sql-map))
    (assoc :select [:*])))

(defn- docs->sql
  [connection table docs]
  (let [doc->sql (get-in connection [:tables table :doc->sql])]
    (into [] (map doc->sql) docs)))

(defn- sql-docs
  [connection table doc-or-docs]
  (docs->sql
   connection table
   (if (map? doc-or-docs)
     [doc-or-docs]
     doc-or-docs)))

(defn- table-create
  [{::keys [connection]} [_ table]]
  (let [columns (get-in connection [:tables table :fields])]
    {:create-table [(sql-name connection table) :if-not-exists]
     :with-columns (into [[:id :text :primary-key]]
                         (mapv (fn [[_key {:keys [sql-name sql-type]}]]
                                 [sql-name sql-type])
                               (dissoc columns :id)))}))

(defn- columns
  [{::keys [connection table]} _]
  {:raw (format "PRAGMA table_info(\"%s\")"
                (name (sql-name connection table)))})

(defn- add-column
  "Bit of a misnomer, as you can only add columns that already appear in the
  table spec, but this is used to ensure a column exists in the db."
  [{::keys [connection table]} [_ column-name]]
  {:alter-table (sql-name connection table)
   :add-column [(sql-name connection table column-name)
                (sql-type connection table column-name)]})

(defn- index-create
  [{::keys [connection]} [_ table column]]
  (let [column-name (double-quote (name (sql-name connection table column)))]
    {:raw (format "CREATE INDEX IF NOT EXISTS %s ON %s (%s)"
                  column-name
                  (double-quote (name (sql-name connection table)))
                  column-name)}))

(defn- table
  [{::keys [query] :as sql-map} [_ table-name]]
  (cond-> (assoc sql-map ::table table-name)
    (and (not (mutate? query))
         (not (count? query)))
    ensure-select))

(defn- get*
  [{::keys [connection table] :as sql-map} [_ id]]
  (assoc sql-map
         :where [:= :id (sql-value connection table :id id)]))

(defn- get-all
  [{::keys [connection table] :as sql-map} [_ ids]]
  (assoc sql-map :where
         [:in :id
          (mapv #(sql-value connection table :id %)
                ids)]))

(defn- upsert
  [{::keys [table connection]} [_ doc-or-docs]]
  (when-let [docs (not-empty (if (map? doc-or-docs)
                               [doc-or-docs]
                               doc-or-docs))]
    (let [fields-map (get-in connection [:tables table :fields])
          field? (fn [key] (contains? fields-map key))]
      (->> docs
           (group-by (comp set keys))
           vals
           (mapv (fn [docs]
                   {:insert-into (sql-name connection table)
                    :values (sql-docs connection table docs)
                    :on-conflict :id
                    :do-update-set {:fields (->> (keys (first docs))
                                                 (filter field?)
                                                 (remove #{:id})
                                                 (map (partial sql-name connection table))
                                                 (filter some?)
                                                 sort
                                                 vec)}}))))))

(defn- update*
  [{::keys [table connection]} [_ doc]]
  (let [doc->sql (get-in connection [:tables table :doc->sql])
        sql-table (sql-name connection table)]
    {:insert-into sql-table
     :values (if-let [serialize-keys (->> (keys doc)
                                          (filter (comp (partial = :json)
                                                     (partial sql-type connection table)))
                                          (map (partial sql-name connection table))
                                          not-empty)]
               [(reduce (fn [doc key]
                          (let [sql-key (sql-name connection table key)]
                            (update doc
                                    sql-key
                                    (fn [json]
                                      {:raw (format (str "json_patch("
                                                         "IFNULL((%s), '{}')"
                                                         ", '%s')")
                                                    (-> {:select [sql-key]
                                                         :from sql-table
                                                         :where [:= :id (:id doc)]}
                                                        (sql/format
                                                         {:quoted true
                                                          :dialect :ansi
                                                          :inline true})
                                                        first)
                                                    (sanitize-string json))}))))
                        (doc->sql doc)
                        serialize-keys)]
               [(doc->sql doc)])
     :on-conflict :id
     :do-update-set {:fields (->> (keys doc)
                                  (remove #{:id})
                                  (map (partial sql-name connection table))
                                  (filter some?)
                                  sort
                                  vec)}}))

(defn- insert
  [{::keys [table connection] :as sql-map} [_ doc-or-docs options]]
  (let [{:keys [conflict] :or {conflict :error}} options]
    (case conflict
      :error {:insert-into (sql-name connection table)
              :values (sql-docs connection table doc-or-docs)}
      :replace {:replace-into (sql-name connection table)
                :values (sql-docs connection table doc-or-docs)}
      :shallow-update (upsert sql-map [:upsert doc-or-docs]))))

(defn- order-by
  [{::keys [table connection] :as sql-map} [_ key-or-tuple]]
  (let [[key direction] (if (coll? key-or-tuple)
                          key-or-tuple
                          [key :asc])]
    (assoc sql-map :order-by
           [[(sql-name connection table key) direction]])))

(defn- limit
  [sql-map [_ limit]]
  (assoc sql-map :limit limit))

(defn- offset
  [sql-map [_ offset]]
  (assoc sql-map :offset offset))

(defn- filter*
  [{::keys [table connection] :as sql-map} [_ filter-map]]
  (->> filter-map
       (map (fn [[key value]]
              [:=
               (sql-name connection table key)
               (sql-value connection table key value)]))
       (concat-where sql-map)))

(defn- gte
  [{::keys [table connection] :as sql-map} [_ key value]]
  (concat-where sql-map
                [[:>=
                  (sql-name connection table key)
                  (sql-value connection table key value)]]))

(defn- gt
  [{::keys [table connection] :as sql-map} [_ key value]]
  (concat-where sql-map
                [[:>
                  (sql-name connection table key)
                  (sql-value connection table key value)]]))

(defn- lte
  [{::keys [table connection] :as sql-map} [_ key value]]
  (concat-where sql-map
                [[:<=
                  (sql-name connection table key)
                  (sql-value connection table key value)]]))

(defn- lt
  [{::keys [table connection] :as sql-map} [_ key value]]
  (concat-where sql-map
                [[:<
                  (sql-name connection table key)
                  (sql-value connection table key value)]]))

(defn- count*
  [{::keys [table connection] :as sql-map} _]
  (cond-> (assoc sql-map :select :%count.*)
    (not (:from sql-map))
    (assoc :from (sql-name connection table))))

(defn- delete*
  [{::keys [table connection] :as sql-map} _]
  (assoc sql-map :delete-from (sql-name connection table)))

(defn- concat-where
  [{:keys [where] :as sql-map} clauses]
  (assoc sql-map :where
         (into (or where [:and])
               clauses)))
