;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns servo-rn.util.sql
  (:require [servo-rn.schema :as s]
            [servo-rn.util.reserved :as r]
            [tempus.core :as t]
            [utilis.inflections :as i]
            [clojure.string :as st]))

(declare throw-on-single-quote
         serialize-date-time
         serialize-boolean
         serialize-keyword
         deserialize-date-time
         deserialize-boolean
         deserialize-keyword)

(defn sanitize-string
  [s]
  (st/replace s "'" "''"))

(defn double-quote
  [s]
  (str "\"" s "\""))

(defn ->sql-name
  [v]
  (when v
    (-> (name v)
        (st/replace #"-" "_")
        throw-on-single-quote
        r/->sql
        keyword)))

(defn sql-name->
  [v]
  (-> (r/sql-> v)
      i/hyphenate
      keyword))

(defn serializer
  [schema]
  (case schema
    :string identity
    :int identity
    :date-time serialize-date-time
    :keyword serialize-keyword
    :boolean serialize-boolean
    (condp = schema
      number? identity
      (s/serializer schema))))

(defn deserializer
  [schema]
  (case schema
    :string identity
    :int identity
    :date-time deserialize-date-time
    :keyword deserialize-keyword
    :boolean deserialize-boolean
    (condp = schema
      number? identity
      (s/deserializer schema))))

(defn ->sql-type
  [schema]
  (case schema
    :string :text
    :int :integer
    :date-time :integer
    :keyword :text
    :boolean :integer
    (condp = schema
      number? :real
      :json)))

;;; Private

(defn- throw-on-single-quote
  [string]
  (when (re-find #"'" (str string))
    (throw
     (js/Error.
      (str "SQL names can not have single quotes: "
           {:name string}))))
  string)

(defn- serialize-date-time
  [value]
  (t/into :long value))

(defn- serialize-keyword
  [value]
  (-> value
      name
      sanitize-string))

(defn- serialize-boolean
  [value]
  (if value
    1
    0))

(defn- deserialize-date-time
  [value]
  (t/from :long value))

(defn- deserialize-keyword
  [value]
  (keyword value))

(defn- deserialize-boolean
  [value]
  (= 1 value))
