;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns servo-rn.connection
  (:require [servo-rn.compile :as c]
            [servo-rn.util.sql :refer [->sql-name ->sql-type double-quote
                                       serializer deserializer]]
            [servo-rn.util.query :refer [op? mutate? count? query->table] :as q]
            [cljs.core.async :as async]
            [cljs.core.async.interop :refer-macros [<p!]]
            [clojure.set :refer [intersection]]
            [servo-rn.schema :as s]
            [fluxus.promise :as p]
            [signum.signal :refer [signal alter!]]
            [react-native-sqlite-2 :default sqlite]
            [utilis.string :refer [format]]
            [utilis.js :as j]
            [clojure.string :as st]))

(declare connect disconnect execute-sql
         create-tables create-indices await-table
         with-table-lookups
         construct-path ensure-some
         desql refresh)

(defn connect
  [{:keys [db-name
           db-version
           db-display-name
           db-max-size-bytes
           db-directory
           on-ready on-error
           tables indices]
    :or {db-version "1.0"
         db-display-name "Servo RN Database"
         db-directory "servo"}
    :as config}]
  (let [{:keys [tables] :as config} (update config :tables with-table-lookups)
        db-name (construct-path db-directory db-name)
        ready-ch (async/chan)
        error (atom nil)
        conn-atom (atom nil)
        ready-handler #(let [ready-conn (assoc @conn-atom :ready-ch nil)]
                         (-> ready-conn
                             (create-tables tables)
                             (j/call :then (fn []
                                             (create-indices ready-conn indices)))
                             (j/call :then (fn []
                                             (async/close! ready-ch)
                                             (when on-ready
                                               (on-ready))))))
        error-handler #(do (reset! error %)
                           (when on-error
                             (on-error %)))
        db (j/call sqlite :openDatabase
                   db-name
                   db-version
                   db-display-name
                   db-max-size-bytes
                   ready-handler
                   error-handler)
        conn (merge config
                    {:db db
                     :db-name db-name
                     :db-version db-version
                     :db-display-name db-display-name
                     :table->callbacks (atom {})
                     :signal->table (atom {})
                     :ready-ch ready-ch
                     :error error})]
    (reset! conn-atom conn)
    conn))

(defn disconnect
  [{:keys [db ready]}]
  (reset! ready false)
  ;; TODO: react-native-sqlite2 doesn't implement a close function
  #_(j/call db :close)
  nil)

(defn run
  [{:keys [ready-ch] :as connection} query]
  (ensure-some connection "No connection provided" {:query query})
  (let [result (p/promise)
        handle-error (fn [error]
                       (js/console.warn
                        (str "Exception occurred running query: "
                             {:query query})
                        error)
                       (p/reject! result error))]
    (async/go
      (when ready-ch
        (async/<! ready-ch))
      (try
        (let [compiled (c/compile connection query)]
          (cond
            (sequential? compiled)
            (let [results (atom [])]
              (doseq [{:keys [sql-expr]} compiled]
                (let [[sql-expr & sql-args] sql-expr
                      raw-result (<p! (execute-sql connection sql-expr sql-args))]
                  (swap! results conj
                         (desql connection query
                                raw-result))))
              (let [results (if (every? coll? @results)
                              (sequence cat @results)
                              @results)
                    results (if (= 1 (count results))
                              (first results)
                              results)]
                (p/resolve! result results))
              (when (mutate? query)
                (refresh connection query)))

            (map? compiled)
            (let [{:keys [sql-expr]} compiled
                  [sql-expr & sql-args] sql-expr]
              (-> connection
                  (execute-sql sql-expr sql-args)
                  (p/then #(try
                             (->> %
                                  (desql connection query)
                                  (p/resolve! result))
                             (when (mutate? query)
                               (refresh connection query))
                             (catch :default e
                               (handle-error e))))
                  (p/catch handle-error)))

            :else (handle-error
                   (js/Error.
                    (str "Unknown compilation result: "
                         compiled)))))
        (catch :default e
          (handle-error e))))
    result))

(defn subscribe
  [connection query]
  (cond
    (mutate? query)
    (throw (js/Error.
            (str "Can not subscribe to a mutating query: "
                 {:query query})))

    query
    (let [signal (signal)
          table (query->table query)
          alter-signal #(alter! signal (constantly %))
          callback (fn []
                     (-> connection
                         (run query)
                         (p/then alter-signal)
                         (p/catch alter-signal))
                     nil)]
      (swap! (:signal->table connection)
             assoc signal table)
      (swap! (:table->callbacks connection)
             assoc-in [table signal]
             {:signal signal
              :callback callback
              :query query
              :ids (or (q/ids query) :any)})
      (callback)
      signal)

    :else (throw (js/Error. "Got nil query in servo-rn/subscribe"))))

(defn dispose
  [connection signal]
  (if-let [table (get @(:signal->table connection) signal)]
    (do (swap! (:signal->table connection) dissoc signal)
        (swap! (:table->callbacks connection)
               update table dissoc signal)
        true)
    (throw (js/Error. (str "Signal is not registered: " signal)))))

;;; Private

(defn- execute-sql
  ([connection sql-expr]
   (execute-sql connection sql-expr nil))
  ([{:keys [db]} sql-expr sql-args]
   (let [result (p/promise)]
     (try (let [sql-args (or (clj->js sql-args) #js [])
                handle-tx (fn [tx]
                            (j/call tx :executeSql
                                    sql-expr sql-args
                                    (fn [_tx raw-result]
                                      (p/resolve! result raw-result))
                                    (fn [_tx e]
                                      (p/reject! result e))))]
            (j/call db :transaction handle-tx))
          (catch :default e
            (p/reject! result e)))
     result)))

(defn- ensure-table
  [connection table-name]
  (let [promise (p/promise)]
    (async/go
      (try (<p! (-> connection
                    (run [[:table-create table-name]])
                    (p/then #(await-table connection table-name))))
           (let [columns (<p! (run connection
                                [[:table table-name]
                                 [:columns]]))]
             (when-let [columns (->> (get-in connection [:tables table-name :field-sql-names])
                                     (remove (set (map (comp keyword :name) columns)))
                                     not-empty)]
               (doseq [column columns]
                 (let [column (name column)
                       clj-name (get-in connection [:tables table-name :sql-name->clj-name column])]
                   (println [:servo-rn.connection/add-column [table-name clj-name]])
                   (<p! (run connection
                          [[:table table-name]
                           [:add-column clj-name]]))))))
           (p/resolve! promise true)
           (catch :default e
             (js/console.warn "Error ensuring table" e)
             (p/reject! promise e))))
    promise))

(defn- await-table
  [connection table-name]
  (let [result (p/promise)
        check-expr (format "SELECT name FROM sqlite_master WHERE type='table' AND name=%s;"
                           (-> table-name
                               ->sql-name
                               name
                               double-quote))]
    (async/go
      (loop []
        (let [raw-result (<p! (execute-sql connection check-expr))]
          (if (pos? (j/get-in raw-result [:rows :length]))
            (p/resolve! result true)
            (do (async/<! (async/timeout 100))
                (recur))))))
    result))

(defn- create-tables
  [conn tables]
  (->> (keys tables)
       (mapv (partial ensure-table conn))
       clj->js
       js/Promise.all))

(defn- create-indices
  [connection indices]
  (->> indices
       (map (fn [[table column]]
              (run connection
                [[:index-create table column]])))
       clj->js
       js/Promise.all))

(defn- valid-table-schema?
  [schema]
  (boolean
   (and (vector? schema)
        (= :map (first schema))
        (every? vector? (rest schema)))))

(defn- compile-doc->sql
  [table schema fields]
  (let [validate (s/validator table schema)]
    (fn [doc]
      (->> (validate doc)
           keys
           (reduce (fn [doc* key]
                     (let [{:keys [value->sql sql-name]} (get fields key)]
                       (if value->sql
                         (let [value (get doc key)]
                           (assoc! doc* sql-name
                                   (when (some? value)
                                     (value->sql value))))
                         (do (js/console.warn
                              (str "Unrecognized key when serializing document: "
                                   {:table table
                                    :doc doc
                                    :key key}))
                             doc*))))
                   (transient {}))
           (persistent!)))))

(defn- compile-sql->doc
  [table sql-name->clj-name fields]
  (fn [doc]
    (when doc
      (->> doc
           js/Object.keys
           (reduce (fn [doc* sql-name]
                     (let [clj-key (sql-name->clj-name sql-name)
                           {:keys [sql->value]} (get fields clj-key)]
                       (if sql->value
                         (let [value (j/get doc sql-name)]
                           (if (some? value)
                             (assoc! doc* clj-key
                                     (when (some? value)
                                       (sql->value value)))
                             doc*))
                         (do (js/console.warn
                              (str "Unrecognized key when deserializing document: "
                                   {:table table
                                    :doc doc
                                    :sql-name sql-name
                                    :sql-name->clj-name sql-name->clj-name}))
                             doc*))))
                   (transient {}))
           (persistent!)))))

(defn- with-table-lookups
  [tables]
  (->> tables
       (map (fn [[table-name {:keys [schema] :as table}]]
              (when (not (valid-table-schema? schema))
                (throw
                 (js/Error.
                  (str "Table schemas must be valid malli map schema: "
                       {:table table-name
                        :schema schema}))))
              (let [fields (->> schema
                                (filter vector?)
                                (map (fn [[key & args]]
                                       (let [schema (last args)]
                                         [key {:sql-name (->sql-name key)
                                               :sql-type (->sql-type schema)
                                               :value->sql (serializer schema)
                                               :sql->value (deserializer schema)
                                               :schema (last args)}])))
                                (into {}))
                    sql-name->clj-name (->> fields
                                            (map (fn [[clj-name {:keys [sql-name]}]]
                                                   [(name sql-name) clj-name]))
                                            (into {}))]
                (when (not (contains? fields :id))
                  (throw
                   (js/Error.
                    (str "All tables must include an :id column which will be used as its primary key: "
                         {:table table-name
                          :schema schema}))))
                [table-name
                 (assoc table
                        :sql-name (->sql-name table-name)
                        :sql-name->clj-name sql-name->clj-name
                        :fields fields
                        :field-sql-names (->> fields
                                              vals
                                              (mapv :sql-name))
                        :sql->doc (compile-sql->doc table-name sql-name->clj-name fields)
                        :doc->sql (compile-doc->sql table-name schema fields))])))
       (into {})))

(defn- refresh
  [connection query]
  (let [table (query->table query)
        mutate-ids (or (q/ids query) :any)]
    (try (let [signals (get @(:table->callbacks connection) table)]
           (doseq [{:keys [callback ids]} (vals signals)]
             (let [refresh? (boolean
                             (or (= :any mutate-ids)
                                 (= :any ids)
                                 (seq (intersection mutate-ids ids))))]
               (when refresh?
                 (callback)))))
         (catch js/Error e
           (js/console.warn
            (str "Exception occurred running query refresh: "
                 {:table table})
            e)))))

(defn- construct-path
  [& components]
  (->> (st/split (st/join "/" components) #"/")
       (remove empty?)
       (st/join "/")))

(defn- ensure-some
  [value message data]
  (when (not value)
    (throw (js/Error. (str message ": " data)))))

(defn- item-seq
  [item row-count tx-fn]
  (loop [i 0
         result (transient [])]
    (if (< i row-count)
      (recur (unchecked-inc i)
             (conj! result (tx-fn (item i))))
      (persistent! result))))

(defn- desql
  [connection query result]
  (if (mutate? query)
    true
    (let [row-count (j/get-in result [:rows :length])]
      (when (pos? row-count)
        (let [table (query->table query)
              item #(j/call-in result [:rows :item] %)
              sql->doc (get-in connection [:tables table :sql->doc])]
          (cond
            (count? query) (-> (item 0)
                               js/Object.values
                               (j/get 0))
            (op? query :get) (sql->doc (item 0))
            (op? query :columns) (item-seq item row-count #(js->clj % :keywordize-keys true))
            :else (item-seq item row-count sql->doc)))))))
