(ns servo-rn.schema
  (:require [tempus.core :refer [DateTime] :as t]
            [malli.core :as m]
            [malli.util :as mu]
            [malli.transform :as mt]
            [utilis.js :as j]
            [clojure.string :as st]))

(def ^:private keyword-prefix "servo/keyword=")
(def ^:private keyword-prefix-pattern (re-pattern keyword-prefix))

(def strip-extra-keys-transformer
  (mt/transformer
   (mt/strip-extra-keys-transformer)))

(defn date-time?
  [x]
  (instance? DateTime x))

(def date-time-transformer
  (mt/transformer
   {:encoders {:date-time {:compile (fn [_ _] {:enter (partial t/into :long)})}}}
   {:decoders {:date-time {:compile (fn [_ _] {:enter (partial t/from :long)})}}}))

(def tx
  (mt/transformer
   (mt/json-transformer)
   date-time-transformer))

(def schema-registry
  (merge
   (m/default-schemas)
   {:date-time (m/-simple-schema
                {:pred date-time?
                 :type :date-time})}))

(defn serialize-keyword
  [kw]
  (str keyword-prefix
       (when-let [n (namespace kw)]
         (str n "/"))
       (name kw)))

(defn deserialize-keyword
  [kw]
  (-> kw
      (st/replace keyword-prefix-pattern "")
      keyword))

(defn serialized-keyword?
  [^String x]
  (j/call x :startsWith keyword-prefix))

(defn serializer
  [schema]
  (let [encode (m/encoder schema {:registry schema-registry} tx)]
    (fn [x]
      (if (and (string? x)
               (re-find keyword-prefix-pattern x))
        (throw (ex-info "Unable to serialize value"
                        {:value x}))
        (cond-> x
          (keyword? x) (serialize-keyword)
          true (-> encode
                   clj->js
                   js/JSON.stringify))))))

(defn deserializer
  [schema]
  (let [decode (m/decoder schema {:registry schema-registry} tx)]
    (fn [x]
      (let [deserialized (-> x
                             js/JSON.parse
                             (js->clj :keywordize-keys true)
                             decode)]
        (cond-> deserialized
          (and (string? deserialized)
               (serialized-keyword? deserialized)) deserialize-keyword)))))

(defn validator
  [table schema]
  (let [schema (mapv #(if (vector? %)
                        (conj (vec (drop-last %))
                              [:maybe (last %)])
                        %)
                     schema)
        schema (mu/optional-keys schema nil {:registry schema-registry})
        validate (m/validator schema {:registry schema-registry})
        explain (m/explainer schema {:registry schema-registry})
        decode (m/decoder schema {:registry schema-registry}
                          strip-extra-keys-transformer)]
    (fn [doc]
      (let [doc (decode doc)]
        (when (not (validate doc))
          (throw
           (js/Error.
            (str "Document did not pass schema validation: "
                 {:table table
                  :doc doc
                  :explain (explain doc)}))))
        doc))))

(def TableCreateQuery
  [:tuple [:= :table-create] :keyword])

(def IndexCreateQuery
  [:tuple [:= :index-create] :keyword :keyword])

(def TableQuery
  [:tuple [:= :table] :keyword])

(def GetQuery
  [:tuple [:= :get] :string])

(def GetAllQuery
  [:tuple [:= :get-all] [:sequential :string]])

(def InsertDoc
  [:map [:id :string]])

(def InsertDocOrDocs
  [:or InsertDoc
   [:sequential
    InsertDoc]])

(def InsertQuery
  [:or
   [:tuple [:= :insert]
    InsertDocOrDocs]
   [:tuple [:= :insert]
    InsertDocOrDocs
    [:map
     [:conflict {:optional true}
      [:enum :error :replace :shallow-update]]]]])

(def UpdateQuery
  [:tuple [:= :update]
   [:map [:id :string]]])

(def OrderByQuery
  [:or
   [:tuple [:= :order-by] [:tuple :keyword [:enum :asc :desc]]]
   [:tuple [:= :order-by] :keyword]])

(def LimitQuery
  [:tuple [:= :limit] pos-int?])

(def OffsetQuery
  [:tuple [:= :offset] :int])

(def FilterQuery
  [:tuple [:= :filter]
   [:map-of :keyword any?]])

(def GteQuery
  [:tuple [:= :gte] :keyword any?])

(def LteQuery
  [:tuple [:= :lte] :keyword any?])

(def GtQuery
  [:tuple [:= :gt] :keyword any?])

(def LtQuery
  [:tuple [:= :lt] :keyword any?])

(def CountQuery
  [:tuple [:= :count]])

(def DeleteQuery
  [:tuple [:= :delete]])

(def ColumnsQuery
  [:tuple [:= :columns]])

(def AddColumnQuery
  [:tuple [:= :add-column] :keyword])

(def Query
  [:vector
   [:multi {:dispatch first}
    [:table-create TableCreateQuery]
    [:index-create IndexCreateQuery]
    [:table TableQuery]
    [:columns ColumnsQuery]
    [:add-column AddColumnQuery]
    [:get GetQuery]
    [:get-all GetAllQuery]
    [:insert InsertQuery]
    [:update UpdateQuery]
    [:order-by OrderByQuery]
    [:limit LimitQuery]
    [:offset OffsetQuery]
    [:filter FilterQuery]
    [:gte GteQuery]
    [:lte LteQuery]
    [:gt GtQuery]
    [:lt LtQuery]
    [:count CountQuery]
    [:delete DeleteQuery]]])

(def query-validator (m/validator Query))
(def query-explainer (m/explainer Query))
