;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns servo.connection
  (:refer-clojure :exclude [send read-string])
  (:require [servo.util.scram :as scram]
            [signum.signal :refer [signal alter!]]
            [tempus.core :as t]
            [vectio.tcp :as tcp]
            [fluxus.flow :as f]
            [fluxus.promise :as p]
            [jsonista.core :as json]
            [utilis.inflections :refer [hyphenate underscore]]
            [utilis.map :refer [map-keys map-vals]]
            [utilis.coll :refer [remove-at insert-at]]
            [utilis.types.number :refer [string->long string->double]]
            [clojure.edn :refer [read-string]]
            [clojure.walk :refer [postwalk]]
            [integrant.core :as ig]
            [spectator.log :refer [error trace]]
            [servo.connection :as servo])
  (:import [tempus.core DateTime]
           [clojure.lang ExceptionInfo MapEntry Namespace]
           [java.nio ByteBuffer ByteOrder]))

(declare connect disconnect ->rql-connection await-tables-ready
         ensure-db ensure-table ensure-index
         json-mapper ->rt-name ->rt-query ->rt-query-term
         send handle-message token request-types)

(defmethod ig/init-key :servo/connection [_ options]
  (connect options))

(defmethod ig/halt-key! :servo/connection [_ connection]
  (try (disconnect connection)
       (catch Exception e
         (error [:servo/connection :disconnect connection] e))))

(defn connect
  [{:keys [db-server db-name await-ready
           response-buffer-size on-disconnected
           tables indices]
    :or {await-ready true
         response-buffer-size 1000}}]
  (f/set-default-on-error! (fn [f e]
                             (error [:servo/connection f] e)))
  (trace [:servo/connection :connect :server db-server])
  (let [{:keys [host port username password]
         :or {host "localhost"
              port 28015
              username "admin"
              password ""}} db-server
        tcp-connection @(tcp/client {:host host :port port})]
    (f/on-close
     tcp-connection
     (fn [_]
       (trace [:servo/connection :tcp-closed :server db-server])))
    @(f/put! tcp-connection (byte-array [0xc3 0xbd 0xc2 0x34]))
    (let [version (try (:server_version (-> @(f/take! tcp-connection)
                                            (json/read-value json/keyword-keys-object-mapper)))
                       (catch Exception _
                         (throw (ex-info ":servo/connection initial handshake failed"
                                         {:db-server db-server :db-name db-name}))))
          send (fn [message]
                 (f/put! tcp-connection
                         (let [json-str (json/write-value-as-string message json-mapper)]
                           ;; Rethink requires JSON strings to be "null terminated"
                           (byte-array (inc (count json-str)) (map int json-str)))))
          recv #(json/read-value @(f/take! tcp-connection) json-mapper)
          {:keys [message client-initial-scram]} (scram/initial-proposal {:username username})
          _ (send message)
          {:keys [server-scram server-authentication]} (scram/server-challenge {:client-initial-scram client-initial-scram}
                                                                               (recv))
          {:keys [message server-signature]} (scram/authentication-request {:client-initial-scram client-initial-scram
                                                                            :server-authentication server-authentication
                                                                            :server-scram server-scram
                                                                            :password password})
          _ (send message)
          {:keys [authentication success]} (recv)]
      (when-not (and success (scram/validate-server {:server-signature server-signature
                                                     :server-authentication authentication}))
        (throw (ex-info ":servo/connection authentication failed"
                        {:client-initial-scram client-initial-scram
                         :server-scram server-scram})))
      (when on-disconnected
        (f/on-close tcp-connection (fn [_] (on-disconnected))))
      @(tcp/add-handler tcp-connection "reql" (tcp/length-based-decoder {:offset 8 :length 4 :max-frame-length (* 5 1024 1024)}))
      (let [connection {:server-version version
                        :db-name db-name
                        :db-term {:db (:query (->rt-query [[:db db-name]]))}
                        :query-counter (atom (long 0))
                        :queries (atom {})
                        :feeds (atom {})
                        :subscriptions (atom {})
                        :tcp-connection tcp-connection
                        :rql-connection (->rql-connection tcp-connection
                                                          db-server)
                        :response-buffer-size response-buffer-size}]

        (f/consume (partial handle-message connection) (:rql-connection connection))
        (ensure-db connection db-name)
        (when (or await-ready
                  (not-empty tables)
                  (not-empty indices))
          (await-tables-ready connection))
        (doseq [table tables]
          (ensure-table connection table))
        (doseq [[table index multi?] indices]
          (ensure-index connection table index :multi (boolean multi?)))
        connection))))

(defn disconnect
  [{:keys [rql-connection tcp-connection] :as connection}]
  (trace [:servo/connection :disconnect connection])
  (f/close! rql-connection)
  (f/close! tcp-connection)
  nil)

(defn on-disconnected
  [{:keys [tcp-connection]} f]
  (f/on-close tcp-connection (fn [_] (f))))

(defonce ^:private var-counter (atom 0))

(defmacro func
  [params body]
  (let [vars (reduce (fn [params p] (assoc params p (swap! var-counter inc))) {} params)]
    `[:func
      [(into [:make-array] ~(mapv vars params))
       (into [~(first body)] ~(mapv (fn [v]
                                      (if-let [vi (get vars v)]
                                        [10 [vi]]
                                        v)) (rest body)))]]))

(defn run
  [{:keys [db-term queries] :as connection} query & {:keys [query-type feed?]
                                                     :or {query-type :start
                                                          feed? (some #(= :changes (first %)) query)}}]
  (let [response-p (p/promise)]
    (try
      (let [token (token connection)
            encoded (->rt-query query)]
        (swap! queries assoc token {:query query
                                    :feed? feed?
                                    :response-p response-p
                                    :response-fn (:response-fn encoded)})
        (when-not @(send connection token [(get request-types query-type) (:query encoded) db-term])
          (swap! queries dissoc token)
          (p/reject! response-p (ex-info ":servo/connection send failed"
                                         {:connection connection :query query :token token})))
        response-p)
      (catch Exception e
        (p/reject! response-p e)))
    response-p))

(defn ->vec
  [s]
  (loop [v (transient [])]
    (let [x @(f/take! s ::none)]
      (if (identical? ::none x)
        (persistent! v)
        (recur (conj! v x))))))

(declare dispose)

(defn subscribe
  [{:keys [feeds subscriptions] :as connection} query]
  (when (some #(= :changes (first %)) query)
    (throw (ex-info ":servo/connection subscribe query should not include :changes" {:query query})))
  (let [sorted? (and (some (comp (partial = :order-by) first) query)
                     (not-any? (comp (partial = :filter) first) query))
        counted? (some (comp (partial = :count) first) query)
        feed @(run connection
                (concat (remove (comp (partial = :count) first) query)
                        [[:changes (merge {:squash true
                                           :include-initial true
                                           :include-types true
                                           :include-states true}
                                          (when sorted? {:include-offsets true}))]])
                :feed? true)
        single (= :atom-feed (get-in @feeds [feed :type]))
        value-signal (with-meta (signal) {:servo/query query})
        initial-value (atom (cond
                              single nil
                              counted? 0
                              :else []))]
    (-> (get-in @feeds [feed :error])
        (p/catch (fn [e] (alter! value-signal (constantly e)))))
    (swap! subscriptions assoc value-signal feed)
    (f/on-close feed (fn [_]
                       (error [:servo/connection :feed-closed
                               :query query]
                              (get-in @feeds [feed :error]))
                       (dispose connection value-signal)))
    (f/consume (fn [change]
                 (try
                   (cond
                     (and (= "state" (:type change)) (= "initializing" (:state change)))
                     nil

                     (and (= "state" (:type change)) (= "ready" (:state change)))
                     (do (alter! value-signal (constantly @initial-value))
                         (reset! initial-value nil))

                     (= "initial" (:type change))
                     (cond
                       single (reset! initial-value (:new-val change))
                       counted? (swap! initial-value inc)
                       :else (swap! initial-value conj (:new-val change)))

                     (= "uninitial" (:type change))
                     (cond
                       single (reset! initial-value nil)
                       counted? (swap! initial-value dec)
                       :else (swap! initial-value (fn [value]
                                                    (let [id (get-in change [:old-val :id])]
                                                      (->> value (remove #(= (:id %) id)) vec)))))

                     (= "add" (:type change))
                     (alter! value-signal
                             (cond
                               single (constantly (:new-val change))
                               counted? inc
                               :else #(if-let [new-offset (:new-offset change)]
                                        (insert-at % new-offset (:new-val change))
                                        (conj % (:new-val change)))))

                     (= "change" (:type change))
                     (alter! value-signal
                             (cond
                               single (constantly (:new-val change))
                               counted? identity
                               :else #(let [{:keys [old-val new-val old-offset new-offset]} change]
                                        (if (and new-offset old-offset)
                                          (-> %
                                              (remove-at old-offset)
                                              (insert-at new-offset new-val))
                                          (mapv (fn [v]
                                                  (if (= (:id v) (:id old-val))
                                                    (:new-val change)
                                                    v))
                                                %)))))

                     (= "remove" (:type change))
                     (alter! value-signal
                             (cond
                               single (constantly nil)
                               counted? dec
                               :else (let [id (get-in change [:old-val :id])]
                                       (comp vec (partial remove #(= (:id %) id))))))

                     :else (throw (ex-info "servo/connection unknown-change"
                                           {:query query
                                            :chage change})))
                   (catch Exception e
                     (error [:servo/connection :query query] e)
                     (alter! value-signal (constantly e)))))
               feed)
    value-signal))

(defn dispose
  [{:keys [subscriptions feeds] :as _connection} value-signal]
  (let [feed (get @subscriptions value-signal)]
    (trace [:servo/connection :signal signal :feed feed
            :query (get-in @feeds [feed :query])
            :token (get-in @feeds [feed :token])])
    (f/close! feed))
  (swap! subscriptions dissoc value-signal)
  (alter! value-signal (constantly :signum.signal/cold)))

(defn noreply-wait
  [connection]
  (run connection nil :query-type :noreply-wait))

(defn server-info
  [connection]
  (run connection nil :query-type :server-info))

(defn ensure-table
  [connection table-name & {:keys [wait]}]
  (try
    @(run connection [[:table-create table-name]])
    (catch ExceptionInfo e
      (let [{:keys [error error-text]} (ex-data e)]
        (when-not (and (= error :op-failed)
                       (= error-text
                          (format "Table `%s.%s` already exists."
                                  (->rt-name (name (:db-name connection)))
                                  (->rt-name (name table-name)))))
          (throw e)))))
  (when wait
    @(run connection [[:table table-name] [:wait]])))

(defn ensure-index
  [connection table-name index-name & {:keys [wait] :as options}]
  (let [options (or (dissoc options :wait) {})]
    (try
      @(run connection [[:table table-name]
                        [:index-create index-name options]])
      (catch ExceptionInfo e
        (let [{:keys [error error-text]} (ex-data e)]
          (when-not (and (= error :op-failed)
                         (= error-text
                            (format "Index `%s` already exists on table `%s.%s`."
                                    (->rt-name (name index-name))
                                    (->rt-name (name (:db-name connection)))
                                    (->rt-name (name table-name)))))
            (throw e)))))
    (when wait
      @(run connection [[:table table-name] [:index-wait index-name]]))))


;;; Private

(def ^:private json-mapper (json/object-mapper
                            {:encode-key-fn (comp underscore name)
                             :decode-key-fn (comp keyword hyphenate)}))

(defn- token
  [{:keys [query-counter]}]
  (swap! query-counter inc))

(defn- ->rql-connection
  [tcp-connection _db-server]
  (let [[rql-connection internal] (f/entangled {:buffer [:fixed 10]} {:buffer [:fixed 10]})]
    (f/consume
     (fn [[token query]]
       (try
         (let [query-bytes (-> query
                               (json/write-value-as-string json-mapper)
                               (.getBytes "UTF-8"))
               size (alength query-bytes)
               message (byte-array (+ 8 4 size))
               bb (.order (ByteBuffer/wrap message) ByteOrder/LITTLE_ENDIAN)]
           (.putLong bb token)
           (.putInt bb size)
           (.put bb query-bytes)
           (f/put! tcp-connection message))
         (catch Exception e
           (error [:servo/connection :send :encode token] e))))
     internal)
    (f/consume
     (fn [bytes]
       (try
         (let [bb (.order (ByteBuffer/wrap bytes) ByteOrder/LITTLE_ENDIAN)
               token (.getLong bb)
               size (.getInt bb)
               message (byte-array size)]
           (.get bb message)
           (f/put! internal [token (json/read-value message json-mapper)]))
         (catch Exception e
           (error [:servo/connection :recv :decode] e))))
     tcp-connection)
    rql-connection))

(defn- send
  [{:keys [rql-connection]} token query]
  (trace [:servo/connection :send :token token :query query])
  (f/put! rql-connection [token query]))

(declare response-types response-note-types response-error-types)

(defn- handle-message
  [{:keys [queries feeds response-buffer-size] :as connection} [token response]]
  (let [{:keys [query response-p response-fn feed?]} (get @queries token)]
    (if-not response-p
      (error [:servo/connection :recv :token token])
      (let [end-query #(swap! queries dissoc token)
            success (partial p/resolve! response-p)
            error (fn [ex]
                    (if-not (p/realized? response-p)
                      (p/reject! response-p ex)
                      (when (f/flow? @response-p)
                        (let [feed @response-p]
                          (swap! feeds update-in [feed :error] #(p/reject! % ex))
                          (f/close! feed)))))]
        (try
          (let [{:keys [r t e n]} response
                handle-seq (fn [& {:keys [close] :or {close false}}]
                             (when-not (and (p/realized? response-p) (f/flow? @response-p))
                               (let [feed (f/flow {:buffer [:fixed response-buffer-size]
                                                   :xform (map response-fn)})]
                                 (f/on-close feed (fn [_]
                                                    (trace [:servo/connection :feed-close :feed feed :query query :token token])
                                                    (end-query)
                                                    (when feed?
                                                      (send connection token [(get request-types :stop)]))
                                                    (swap! feeds dissoc feed)))
                                 (swap! feeds assoc feed {:type (get response-note-types (first n))
                                                          :query query
                                                          :token token
                                                          :error (p/promise)})
                                 (success feed)))
                             (let [feed @response-p]
                               (-> feed
                                   (f/put-all! r)
                                   (p/finally (fn [_]
                                                (if close
                                                  (f/close! feed)
                                                  (send connection token [(get request-types :continue)])))))))]
            (trace [:servo/connection :response :query query :token token
                    :response response :response-type (get response-types t)])
            (case (get response-types t)
              :success-atom
              (do (success (response-fn (first r)))
                  (end-query))

              :success-sequence
              (handle-seq :close true)

              :success-partial
              (handle-seq :close false)

              :wait-complete
              (do (success (response-fn nil))
                  (end-query))

              :server-info
              (do (success (response-fn (first r)))
                  (end-query))

              :client-error
              (do (error (ex-info ":servo/connection client-error"
                                  {:query query :error (get response-error-types e) :error-text (first r)}))
                  (end-query))

              :compile-error
              (do (error (ex-info ":servo/connection compile-error"
                                  {:query query :error (get response-error-types e) :error-text (first r)}))
                  (end-query))

              :runtime-error
              (do (error (ex-info ":servo/connection runtime-error"
                                  {:query query :error (get response-error-types e) :error-text (first r)}))
                  (end-query))))
          (catch Exception e
            (error (ex-info ":servo/connection response"
                            {:query query :token token :response response :exception e}))
            (end-query)))))))

(declare term-types response-types response-error-types
         ->rt ->rt-options ->rt-key ->rt-name ->rt-value
         rt-> rt-string-> rt-key-> rt-name-> rt-value->)

(defn- walk
  [kf vf v]
  (cond
    (instance? DateTime v) (vf v)
    (and (map? v) (= "TIME" (get v :$reql-type$))) (vf v)
    (map-entry? v) (try
                     (MapEntry/create
                      (kf (key v))
                      (walk kf vf (val v)))
                     (catch ExceptionInfo e
                       (if (= :invalid-value (:cause (ex-data e)))
                         (throw (ex-info ":servo/connection invalid-value"
                                         {:key (key v) :value (val v)}))
                         (throw e))))
    (list? v) (apply list (map (partial walk kf vf) v))
    (seq? v) (doall (map (partial walk kf vf) v))
    (coll? v) (into (empty v) (map (partial walk kf vf) v))
    :else (vf v)))

(defn- ->rt-query-term
  [[id & parameters]]
  (merge
   {:id (get term-types id)}
   (let [nested-fields #(reduce merge
                                ((fn xform [params]
                                   (map (fn [v]
                                          (cond
                                            (or (keyword? v) (string? v)) {(->rt-key v) true}
                                            (map? v) (->> (map (fn [[k v]]
                                                                 [(->rt-key k) (reduce merge (xform v))]) v)
                                                          (into {}))))
                                        params)) parameters))]
     (case id
       :db-list {:response-fn (partial map rt-name->)}
       :db {:arguments [(->rt-name (first parameters))]}
       :db-create {:arguments [(->rt-name (first parameters))]}
       :table-list {:response-fn (partial map rt-name->)}
       :index-list {:response-fn (partial map rt-name->)}
       :table {:arguments [(->rt-name (first parameters))]
               :response-fn rt->}
       :table-create {:arguments [(->rt-name (first parameters))]}
       :table-drop {:arguments [(->rt-name (first parameters))]}
       :index-create {:arguments [(->rt-name (first parameters))]
                      :options (->rt (second parameters))}
       :index-wait {:arguments [(->rt-name (first parameters))]}
       :index-rename {:arguments [map] ->rt-name parameters}
       :index-drop {:arguments [(->rt-name (first parameters))]}
       :get {:arguments [(if-some [id (first parameters)]
                           (->rt-value id)
                           (throw (ex-info ":servo/connection :get nil id"
                                           {:parameters parameters})))]
             :response-fn rt->}
       :get-all {:arguments (let [keys (first parameters)]
                              (if (vector? keys)
                                (mapv ->rt-value keys)
                                [(->rt-value keys)]))
                 :options (when-let [index (second parameters)]
                            {"index" (->rt-name (:index index))})
                 :response-fn rt->}
       :insert (let [[value options] parameters]
                 {:arguments [(if (or (map? value) (not (coll? value)))
                                (->rt value)
                                [(get term-types :make-array)
                                 (mapv ->rt value)])]
                  :options (->rt-options options)})
       :update (let [[value options] parameters]
                 {:arguments [(->rt value)]
                  :options (->rt-options options)})
       :filter (let [[value options] parameters]
                 (merge
                  (cond
                    (map? value)
                    {:arguments [(->rt value)]})
                  {:options options
                   :response-fn rt->}))
       :between (let [[lower upper options] parameters]
                  {:arguments (mapv ->rt-value [lower upper])
                   :options (not-empty
                             (merge
                              (when-let [index (:index options)]
                                {"index" (->rt-name index)})
                              (when-let [left (:left-bound options)]
                                {"left_bound" (name left)})
                              (when-let [right (:right-bound options)]
                                {"right_bound" (name right)})))})
       :order-by (let [[param] parameters
                       ->rt-sort #(if (and (vector? %)
                                           (#{:asc :desc} (first %)))
                                    (let [[order index] %]
                                      [(get term-types order)
                                       [(->rt-name index)]])
                                    (->rt-name %))]
                   (if (map? param)
                     (let [{:keys [index]} param]
                       {:options {"index" (->rt-sort index)}})
                     {:arguments [(->rt-sort param)]}))
       :get-field {:arguments [(->rt-name (first parameters))]}
       :group {:arguments (let [fields (first parameters)]
                            (if (vector? fields)
                              (mapv ->rt-name fields)
                              [(->rt-name fields)]))
               :response-fn (fn [{:keys [data]}]
                              (reduce (fn [groups [group-id docs]]
                                        (assoc groups (rt-> group-id) (mapv rt-> docs))) {} data))}
       :max {:arguments [(->rt-name (first parameters))]}
       :min {:arguments [(->rt-name (first parameters))]}
       :pluck {:arguments [(nested-fields)]}
       :with-fields {:arguments [(nested-fields)]}
       :without {:arguments [(nested-fields)]}
       :contains {:arguments [(first (->rt-value (first parameters)))]}
       :nth {:arguments [(first parameters)]}
       ;;:pred (pred (first parameters))
       :distinct (when-some [index (first parameters)]
                   {:options {"index" (->rt-name index)}})
       :slice (let [[start opt1 opt2] parameters
                    [end opts] (if (map? opt1)
                                 [nil opt1]
                                 [opt1 opt2])]
                {:arguments (vec (concat [start] (when end [end])))
                 :options (let [{:keys [left-bound right-bound]} opts]
                            (merge (when left-bound
                                     {"left_bound" (->rt-name left-bound)})
                                   (when right-bound
                                     {"right_bound" (->rt-name right-bound)})))})
       :during (let [[start end] parameters]
                 {:arguments [[(->rt-value start) (->rt-value end)]]})
       ;; :func (let [[params body] (first parameters)]
       ;;         [(->rt-query-term params) (->rt-query-term body)])
       :changes {:options (map-keys (comp underscore name) (first parameters))
                 :response-fn #(cond-> % (:new-val %) (update :new-val rt->) (:old-val %) (update :old-val rt->))}
       {:arguments parameters}))))

(defn- ->rt-query
  [query]
  {:query (reduce (fn [query term]
                    (let [{:keys [id arguments options]} (->rt-query-term term)]
                      (remove nil? [id (cond->> arguments query (cons query)
                                                true (remove nil?)) options]))) nil query)
   :response-fn (or (->> (map (comp :response-fn ->rt-query-term) query)
                         (remove nil?)
                         last)
                    rt->)})

(defn- namespaced-string
  [k]
  (if (keyword? k)
    (if-let [ns (namespace k)]
      (str ns "/" (name k))
      (name k))
    k))

(defn- ->rt-name
  [s]
  (underscore (namespaced-string s)))

(defn- ->rt-key
  [k]
  (cond
    (keyword? k) (->rt-name k)
    (string? k) (str "servo/str=" (->rt-name k))
    (double? k) (str "servo/double=" (str k))
    (int? k) (str "servo/long=" (str k))
    :else (throw (ex-info ":servo/connection unsupported type for key" {:key k}))))

(defn- decode-servo-string
  [s]
  (re-find #"^servo/(.*)=(.*)$" s))

(defn- ->rt-value
  [v]
  (try
    (cond
      (or (number? v) (boolean? v)) v
      (string? v) (if (decode-servo-string v)
                    (throw (Exception. "invalid-value"))
                    v)
      (keyword? v) (str "servo/keyword=" (->rt-name v))
      (instance? DateTime v) {"$reql_type$" "TIME"
                              "epoch_time" (double (/ (t/into :long v) 1000))
                              "timezone" "+00:00"}
      :else (let [encoded (pr-str v)]
              (if (= v (read-string {:readers *data-readers*}
                                    encoded))
                (str "servo/edn=" encoded)
                (throw (Exception. "invalid-value")))))
    (catch Exception _
      (throw (ex-info ":servo/connection invalid-value"
                      {:cause :invalid-value :value v})))))

(defn- ->rt
  [v]
  (->> (walk ->rt-key ->rt-value v)
       (postwalk (fn [v]
                   (if (and (not (map? v))
                            (not (map-entry? v))
                            (or (list? v)
                                (vector? v)
                                (seq? v)))
                     [(get term-types :make-array) v]
                     v)))))

(defn- ->rt-options
  [options]
  (->> options
       (map-vals (fn [option]
                   (if (keyword? option)
                     (name option)
                     option)))
       ->rt))

(defn- rt-name->
  [s]
  (keyword (hyphenate s)))

(defn- rt-key->
  [k]
  (let [k (namespaced-string k)
        coerced (rt-string-> k)]
    (if (= k coerced)
      (keyword (hyphenate k))
      coerced)))

(defn- rt-string->
  [s]
  (let [[_ type-fn value-str] (decode-servo-string s)]
    (try
      (if (and type-fn value-str)
        ((case type-fn
           "keyword" (comp keyword hyphenate)
           "str" str
           "double" string->double
           "long" string->long
           "edn" (partial read-string {:readers *data-readers*}))
         value-str)
        s)
      (catch Exception e
        (error [:servo/connection :rt-string-> :string s] e)))))

(defn- rt-value->
  [v]
  (cond
    (and (map? v) (= "TIME" (get v :$reql-type$))) (t/from :long (* 1000 (get v :epoch-time)))
    (instance? MapEntry v) (MapEntry/create (rt-key-> (key v)) (val v))
    (string? v) (rt-string-> v)
    :else v))

(defn- rt->
  [m]
  (walk rt-key-> rt-value-> m))

(defn- ensure-db
  [connection db-name]
  (try
    @(run connection [[:db-create db-name]])
    (catch ExceptionInfo e
      (let [{:keys [error error-text]} (ex-data e)]
        (when-not (and (= error :op-failed)
                       (= error-text
                          (format "Database `%s` already exists."
                                  (->rt-name (name db-name)))))
          (throw e))))))

(defn- await-tables-ready
  [connection]
  (let [table-list @(run connection [[:table-list]])
        await-tables-ready (fn []
                             (->> table-list
                                  (map (fn [table]
                                         @(run connection [[:table table] [:status]])))
                                  (filter (comp :all-replicas-ready :status))
                                  count))]
    (loop []
      (let [ready-count (await-tables-ready)]
        (when (< ready-count (count table-list))
          (trace [::await-tables ::ready ready-count ::total (count table-list)])
          (Thread/sleep 1000)
          (recur))))
    (trace [::await-tables ::ready (count table-list) ::total (count table-list)])))

(def ^:private request-types
  {:start 1
   :continue 2
   :stop 3
   :noreply-wait 4
   :server-info 5})

(def ^:private response-types
  {1 :success-atom
   2 :success-sequence
   3 :success-partial
   4 :wait-complete
   5 :server-info
   16 :client-error
   17 :compile-error
   18 :runtime-error})

(def ^:private response-error-types
  {1000000 :internal
   2000000 :resource-limit
   3000000 :query-logic
   3100000 :non-existence
   4100000 :op-failed
   4200000 :op-indeterminate
   5000000 :user
   6000000 :permission-error})

(def ^:private response-note-types
  {1 :sequence-feed
   2 :atom-feed
   3 :order-by-limit-feed
   4 :unioned-feed
   5 :includes-states})

(def ^:private term-types
  {:datum 1
   :make-array 2
   :make-obj 3
   :javascript 11
   :uuid 169
   :http 153
   :error 12
   :implicit-var 13
   :db 14
   :table 15
   :get 16
   :get-all 78
   :eq 17
   :ne 18
   :lt 19
   :le 20
   :gt 21
   :ge 22
   :not 23
   :add 24
   :sub 25
   :mul 26
   :div 27
   :mod 28
   :floor 183
   :ceil 184
   :round 185
   :append 29
   :prepend 80
   :difference 95
   :set-insert 88
   :set-intersection 89
   :set-union 90
   :set-difference 91
   :slice 30
   :skip 70
   :limit 71
   :offsets-of 87
   :contains 93
   :get-field 31
   :keys 94
   :values 186
   :object 143
   :has-fields 32
   :with-fields 96
   :pluck 33
   :without 34
   :merge 35
   :between-deprecated 36
   :between 182
   :reduce 37
   :map 38
   :fold 187
   :filter 39
   :concat-map 40
   :order-by 41
   :distinct 42
   :count 43
   :is-empty 86
   :union 44
   :nth 45
   :bracket 170
   :inner-join 48
   :outer-join 49
   :eq-join 50
   :zip 72
   :range 173
   :insert-at 82
   :delete-at 83
   :change-at 84
   :splice-at 85
   :coerce-to 51
   :type-of 52
   :update 53
   :delete 54
   :replace 55
   :insert 56
   :db-create 57
   :db-drop 58
   :db-list 59
   :table-create 60
   :table-drop 61
   :table-list 62
   :config 174
   :status 175
   :wait 177
   :reconfigure 176
   :rebalance 179
   :sync 138
   :grant 188
   :index-create 75
   :index-drop 76
   :index-list 77
   :index-status 139
   :index-wait 140
   :index-rename 156
   :set-write-hook 189
   :get-write-hook 190
   :funcall 64
   :branch 65
   :or 66
   :and 67
   :for-each 68
   :func 69
   :asc 73
   :desc 74
   :info 79
   :match 97
   :upcase 141
   :downcase 142
   :sample 81
   :default 92
   :json 98
   :iso8601 99
   :to-iso8601 100
   :epoch-time 101
   :to-epoch-time 102
   :now 103
   :in-timezone 104
   :during 105
   :date 106
   :time-of-day 126
   :timezone 127
   :year 128
   :month 129
   :day 130
   :day-of-week 131
   :day-of-year 132
   :hours 133
   :minutes 134
   :seconds 135
   :time 136
   :monday 107
   :tuesday 108
   :wednesday 109
   :thursday 110
   :friday 111
   :saturday 112
   :sunday 113
   :january 114
   :february 115
   :march 116
   :april 117
   :may 118
   :june 119
   :july 120
   :august 121
   :september 122
   :october 123
   :november 124
   :december 125
   :literal 137
   :group 144
   :sum 145
   :avg 146
   :min 147
   :max 148
   :split 149
   :ungroup 150
   :random 151
   :changes 152
   :arguments 154
   :binary 155
   :geojson 157
   :to-geojson 158
   :point 159
   :line 160
   :polygon 161
   :distance 162
   :intersects 163
   :includes 164
   :circle 165
   :get-intersecting 166
   :fill 167
   :get-nearest 168
   :polygon-sub 171
   :to-json-string 172
   :minval 180
   :maxval 181
   :bit-and 191
   :bit-or 192
   :bit-xor 193
   :bit-not 194
   :bit-sal 195
   :bit-sar 196})

;;; Force loading of all data reader functions
(doseq [v (vals *data-readers*)]
  (let [^Namespace n (->> v meta :ns)]
    (require (.getName n))))
