;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns signum.debounce
  (:require
   [signum.events :as se]
   [signum.fx :as fx]
   [utilis.timer :as t]))

(def ^:private deferred-actions (atom {}))

(defn dispatch-debounce
  [coeffects dispatch-map]
  (let [cancel-timeout (fn [id]
                         (when-let [deferred (get @deferred-actions id)]
                           (t/cancel (:timer deferred))
                           (swap! deferred-actions dissoc id)))
        run-action (fn [action event]
                     (cond
                       (= :dispatch action) (se/dispatch coeffects event)
                       (= :dispatch-n action) (doseq [e event]
                                                (se/dispatch coeffects e))))
        {:keys [id timeout action event]} dispatch-map]
    (cond
      (#{:dispatch :dispatch-n} action)
      (do (cancel-timeout id)
          (swap! deferred-actions assoc id
                 {:action action
                  :event event
                  :timer (t/run-after
                          (fn []
                            (cancel-timeout id)
                            (run-action action event))
                          timeout)}))

      (= :cancel action)
      (cancel-timeout id)

      (= :flush action)
      (when-let [{:keys [action event]} (get @deferred-actions id)]
        (cancel-timeout id)
        (run-action action event))

      :else
      (throw (ex-info (str ":signum/dispatch-debounce invalid action " action) dispatch-map)))))

(fx/reg-fx :dispatch-debounce dispatch-debounce)
