;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns signum.events
  (:refer-clojure :exclude [namespace])
  (:require
   [Signum.fx :as fx]
   [signum.interceptors :as interceptors]))

(defonce handlers (atom {}))

(declare handler-interceptor)

(defn reg-event
  ([id handler]
   (reg-event id nil handler))
  ([id interceptors handler]
   (swap! handlers assoc id {:queue (concat [fx/interceptor]
                                            interceptors
                                            [(handler-interceptor id handler)])
                             :stack []
                             :ns *ns*})
   id))

(defn dispatch
  [coeffects [event-id & _ :as event-v]]
  (if-let [handler-context (get @handlers event-id)]
    (-> handler-context
        (update :coeffects merge coeffects)
        (assoc :event-v event-v)
        interceptors/run)
    (throw (ex-info ":signum/dispatch unhandled event" {:event-v event-v}))))

(fx/reg-fx
 :signum/dispatch
 (fn [coeffects event-v]
   (dispatch coeffects event-v)))

(defn namespace
  [id]
  (:ns (get @handlers id)))

(defn event?
  [event-id]
  (contains? @handlers event-id))

(defn events
  []
  (keys @handlers))

(defn trace
  [event-id trace-id trace-fn]
  (swap! handlers assoc-in [event-id :tracers trace-id] trace-fn)
  event-id)

(defn detrace
  [event-id trace-id]
  (swap! handlers update-in [event-id :tracers] dissoc trace-id)
  event-id)


;;; Private

(defn- handler-interceptor
  [id handler-fn]
  (interceptors/->interceptor
   :id id
   :before (fn [{:keys [coeffects event-v] :as context}]
             (let [effects (handler-fn coeffects event-v)]
               (when-let [tracers (not-empty (get-in @handlers [id :tracers]))]
                 (doseq [[trace-id trace-fn] tracers]
                   (trace-fn trace-id event-v :dispatch effects)))
               (update context :effects merge (when (map? effects) effects))))))
