;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns signum.subs
  (:refer-clojure :exclude [namespace subs])
  ;;#?(:cljs (:require-macros [signum.signal :refer [with-tracking when-hot]]))
  (:require [signum.signal :refer [with-tracking when-hot] :as s]
            [signum.interceptors :refer [->interceptor] :as interceptors]
            [signum.fx :as fx]
            [utilis.fn :refer [fsafe]]
            [utilis.map :refer [compact]]
            [utilis.timer :as ut]
            #?(:cljs [utilis.js :as j])
            [clojure.core.async :refer [dropping-buffer chan close!
                                        #?@(:clj [thread <!! >!!]
                                            :cljs [go <! >!])]]
            [clojure.set :as set]
            [integrant.core :as ig])
  #?(:clj (:import [signum.signal Signal]
                   [clojure.lang ExceptionInfo])))

(defonce ^:dynamic *context* {})
(defonce ^:dynamic *current-sub-fn* ::computation-fn)
(defonce ^:private limbo-timer-ms (atom (* 15 60 1000))) ; 15 minutes

(defonce ^:private logging-lock #?(:clj (Object.) :cljs (js/Object.)))

(declare handlers reset-subscriptions! release! signal-interceptor)

(defmethod ig/init-key :signum/subs
  [_ {:keys [limbo-time-ms]}]
  (reset! limbo-timer-ms limbo-time-ms))

(defn reg-sub
  [query-id & args]
  (when-not (fn? (last args))
    (throw (ex-info "computation-fn must be provided"
                    {:query-id query-id :args args})))
  (let [interceptors (when (vector? (first args)) (first args))
        fns (filter fn? args)
        [init-fn dispose-fn computation-fn] (case (count fns)
                                              3 fns
                                              2 [(first fns) nil (last fns)]
                                              1 [nil nil (last fns)])]
    (swap! handlers assoc query-id
           (compact
            {:init-fn init-fn
             :dispose-fn dispose-fn
             :computation-fn computation-fn
             :queue (vec (concat [fx/interceptor]
                                 interceptors
                                 [signal-interceptor]))
             :stack []
             :ns *ns*})))
  (reset-subscriptions! query-id)
  query-id)

(defn subscribe
  [[query-id & _ :as query-v]]
  (when (= ::init-fn *current-sub-fn*)
    (throw (ex-info "subscribe is not supported within the init-fn of a sub"
                    {:query-v query-v})))
  (when (= ::dispose-fn *current-sub-fn*)
    (throw (ex-info "subscribe is not supported within the dispose-fn of a sub"
                    {:query-v query-v})))
  (if-let [handler-context (get @handlers query-id)]
    (-> (merge *context* handler-context)
        (assoc-in [:coeffects :query-v] query-v)
        interceptors/run
        (get-in [:effects :signal]))
    (throw (ex-info "invalid query" {:query query-v}))))

(defn interceptors
  [query-id]
  (get-in @handlers [query-id :queue]))

(defn namespace
  [id]
  (:ns (get @handlers id)))

(declare signals)

(defn sub?
  [id]
  (contains? @handlers id))

(defn subs
  []
  (keys @signals))


;;; Private

(defonce ^:private handlers (atom {}))      ; key: query-id
(defonce ^:private signals (atom {}))       ; key: query-v
(defonce ^:private subscriptions (atom {})) ; key: output-signal

(defn- create-subscription!
  [query-v output-signal]
  (let [handlers (get @handlers (first query-v))
        {:keys [init-fn computation-fn]} handlers
        init-context (when init-fn
                       (binding [*current-sub-fn* ::init-fn]
                         (try
                           (init-fn query-v)
                           (catch #?(:clj Exception :cljs js/Error) e
                             (s/alter! output-signal (constantly e))))))
        input-signals (atom #{})
        compute-ch (chan (dropping-buffer 1))]
    (#?(:clj thread :cljs go)
     (loop []
       (when-let [compute (#?(:clj <!! :cljs <!) compute-ch)]
         (try
           (reset! (:computing (meta output-signal)) true)
           (binding [*current-sub-fn* ::compute-fn]
             (let [derefed (atom #{})]
               (with-tracking
                 (fn [reason s]
                   (when (= :deref reason)
                     (when-not (or (get @input-signals s)
                                   (get @derefed s))
                       (swap! input-signals conj s)
                       (add-watch s query-v
                                  (fn [_ _ old-value new-value]
                                    (when (not= old-value new-value)
                                      #?(:clj (>!! compute-ch {:reason :watch :signal s})
                                         :cljs (go (>! compute-ch {:reason :watch :signal s})))))))
                     (swap! derefed conj s)))
                 (let [value (try
                               (if init-fn
                                 (computation-fn init-context query-v)
                                 (computation-fn query-v))
                               (catch ExceptionInfo e
                                 (if (= :signum.signal/cold (:cause (ex-data e)))
                                   :signum.signal/cold
                                   (throw e))))]
                   (doseq [w (set/difference @input-signals @derefed)]
                     (remove-watch w query-v))
                   (reset! input-signals @derefed)
                   (s/alter! output-signal (constantly value))))))
           (catch #?(:clj Exception :cljs js/Error) e
             (locking logging-lock
               (#?(:clj println :cljs js/console.error)
                (str ":signum.subs/compute-fn " (pr-str query-v) " error\n") e))
             e)
           (finally (reset! (:computing (meta output-signal)) false)))
         (recur))))
    #?(:clj (>!! compute-ch {:reason :initial})
       :cljs (go (>! compute-ch {:reason :initial})))
    (swap! subscriptions assoc output-signal (compact
                                              {:query-v query-v
                                               :context *context*
                                               :handlers handlers
                                               :init-context init-context
                                               :input-signals input-signals
                                               :compute-ch compute-ch}))))

(defn- dispose-subscription!
  [query-v output-signal]
  (when-let [subscription (get @subscriptions output-signal)]
    (let [{:keys [init-context handlers input-signals compute-ch]} subscription]
      (close! compute-ch)
      (doseq [w @input-signals] (remove-watch w query-v))
      (when-let [dispose-fn (:dispose-fn handlers)]
        (binding [*current-sub-fn* ::dispose-fn]
          (try
            (dispose-fn init-context query-v)
            (s/alter! output-signal (constantly :signum.signal/cold))
            (catch #?(:clj Exception :cljs js/Error) e
              (s/alter! output-signal (constantly e))))))
      (swap! subscriptions dissoc output-signal)
      nil)))

(defn- reset-subscriptions!
  [query-id]
  (locking signals
    (doseq [[query-v output-signal] (filter (fn [[query-v _]] (= query-id (first query-v))) @signals)]
      (let [output-signal (:signal output-signal)
            context (get-in @subscriptions [output-signal :context])]
        (dispose-subscription! query-v output-signal)
        (binding [*context* context]
          (create-subscription! query-v output-signal))))))

(defn- handle-watches
  [query-v output-signal _ _ _old-watchers watchers]
  (locking signals
    (if (zero? (count watchers))
      (swap! signals update query-v assoc :limbo
             (ut/run-after #(do
                              (swap! signals dissoc query-v)
                              (dispose-subscription! query-v output-signal))
                           @limbo-timer-ms))
      (when-not (get @subscriptions output-signal)
        (create-subscription! query-v output-signal)))))

(defn- signal
  [query-v]
  (locking signals
    (or (let [signal (get @signals query-v)]
          (when-let [timer (:limbo signal)]
            (ut/cancel timer)
            (swap! signals update query-v dissoc :limbo))
          (:signal signal))
        (let [output-signal (with-meta (s/signal) {:query-v query-v
                                                   :computing (atom false)})]
          (s/add-watcher-watch output-signal
                               query-v
                               (partial handle-watches query-v output-signal))
          (swap! signals assoc-in [query-v :signal] output-signal)
          output-signal))))

(def ^:private signal-interceptor
  (->interceptor
   :id :signum.subs/signal-interceptor
   :before #(assoc-in % [:effects :signal] (signal (get-in % [:coeffects :query-v])))))
