;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns signum.signal
  #?(:cljs (:require-macros [signum.signal]))
  (:require #?(:cljs [signum.SignalColdException :refer [SignalColdException]])
            [utilis.fn :refer [fsafe]]
            [utilis.string :as ust]
            #?(:cljs [utilis.js :as j]))
  #?(:clj (:import [signum SignalColdException]
                   [clojure.lang IRef IDeref IObj IMeta]
                   [java.util.concurrent Executors]
                   [java.lang Thread])))

#?(:clj
   (def ^:private executor
     (Executors/newThreadPerTaskExecutor
      (-> (Thread/ofVirtual)
          (.name "signum-" 0)
          (.factory)))))
(def ^:dynamic *tracker* nil)

(declare pr-signal)

(defprotocol IWatchWatchers
  (add-watcher-watch [signal key watch-fn])
  (remove-watcher-watch [signal key])
  (watchers [signal]))

(defprotocol IState
  (hot? [signal])
  (cold? [signal])
  (error? [signal]))

(defprotocol IAlter
  (-alter! [signal fun args]))

(defn alter! [signal fun & args] (-alter! signal fun args))

(deftype Signal [backend watches meta-map]

  Object
  (toString [^Signal this]
    (pr-signal this))
  #?(:cljs IHash)
  (#?(:clj hashCode :cljs -hash) [_]
    (hash [:signum/signal backend]))
  #?(:cljs IEquiv)
  (#?(:clj equals :cljs -equiv) [this other]
    (= (#?(:clj .backend :cljs .-backend) this)
       (#?(:clj .backend :cljs .-backend) ^Signal other)))

  IState
  (hot? [this]
    (not (or (cold? this) (error? this))))
  (cold? [_this]
    (= :signum.signal/cold (clojure.core/deref backend)))
  (error? [_this]
    (instance? #?(:clj Throwable :cljs js/Error)
               (clojure.core/deref backend)))

  IDeref
  (#?(:clj deref :cljs -deref)
    [this]
    ((fsafe *tracker*) :deref this)
    (let [value (clojure.core/deref backend)]
      (if (= :signum.signal/cold value)
        (throw (SignalColdException. {:signal this} nil))
        (cond-> value (instance? #?(:clj Throwable :cljs js/Error) value) throw))))

  IAlter
  (-alter!
    [this fun args]
    (let [mutation-fn (fn [old-value]
                        (try
                          (apply fun old-value args)
                          (catch #?(:clj Throwable :cljs :default) e e)))]
      (#?@(:clj [send-via executor] :cljs [swap!]) backend mutation-fn))
    this)

  #?(:clj IRef :cljs IWatchable)
  (#?(:clj addWatch :cljs -add-watch)
    [this watch-key watch-fn]
    (add-watch
     backend watch-key
     (fn [_key _ref old-value new-value]
       (when (not= old-value new-value)
         (watch-fn watch-key this old-value new-value))))
    (swap! watches assoc watch-key watch-fn)
    (when-not (cold? this)
      (let [value (clojure.core/deref backend)]
        (watch-fn watch-key this :signum.signal/initial value)))
    this)
  (#?(:clj removeWatch :cljs -remove-watch)
    [this watch-key]
    (swap! watches dissoc watch-key)
    (remove-watch backend watch-key)
    this)

  #?(:clj IObj :cljs IWithMeta)
  (#?(:clj withMeta :cljs -with-meta)
    [_ meta-map]
    (Signal. backend watches meta-map))

  IMeta
  (#?(:clj meta :cljs -meta)
    [_]
    meta-map)

  #?@(:cljs
      [IPrintWithWriter
       (-pr-writer [this w _opts] (write-all w (pr-signal this)))])

  IWatchWatchers
  (add-watcher-watch [this watch-key watch-fn]
    (add-watch watches watch-key (fn [key _ref old-value new-value]
                                   (watch-fn key this old-value new-value))))
  (remove-watcher-watch [_this watch-key]
    (remove-watch watches watch-key)))

#?(:clj
   (defmethod print-method Signal [^Signal s w]
     (.write ^java.io.Writer w ^String (pr-signal s))))

(defn signal
  ([] (signal :signum.signal/cold))
  ([state]
   (let [s (Signal. (#?(:clj agent :cljs atom) state) (atom {}) nil)]
     ((fsafe *tracker*) :create s)
     s)))

#?(:clj
   (defmacro with-tracking
     [tracker-fn & body]
     `(binding [*tracker* ~tracker-fn]
        ~@body)))


;;; Private

(defn- pr-signal
  [^Signal signal]
  (ust/format (str "#<signum/signal@" #?(:clj "0x%x" :cljs "%s") ": %s>") (hash signal)
              (try (pr-str #?(:clj @(.backend signal) :cljs @(j/get signal :backend)))
                   #?(:clj  (catch Throwable e (.getMessage e))
                      :cljs (catch js/Error e (j/get e :message))))))
