;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns spectator-slf4j.adapter
  (:gen-class :name com.7theta.spectator.slf4j.SpectatorLoggerAdapter
              :implements [org.slf4j.spi.LocationAwareLogger]
              :state state
              :init init
              :constructors {[String] []})
  (:require [spectator.log :as log]
            [clojure.string :as st])
  (:import  [com.7theta.spectator.slf4j SpectatorLoggerAdapter]
            org.slf4j.Marker
            org.slf4j.helpers.MessageFormatter
            org.slf4j.spi.LocationAwareLogger))

(defn -init
  [logger-name]
  (log/trace [:spectator.slf4j/logger :create logger-name])
  [[] logger-name])

(defn -getName
  [^SpectatorLoggerAdapter this]
  (.state this))

(declare identify-caller)

(defmacro define-methods
  [method-name level]
  `(do
     ~@(for [signature ["-String" "-String-Object" "-String-Object-Object" "-String-Object<>" "-String-Throwable"]
             with-marker? [false true]]
         (let [func-sym (symbol (str method-name (when with-marker? "-Marker") signature))
               args-sym (gensym "args")
               file-sym (gensym "file")
               line-sym (gensym "line")]
           `(defn ~func-sym [^SpectatorLoggerAdapter this# & ~args-sym]
              (let [stack# (.getStackTrace (Thread/currentThread))
                    ^StackTraceElement caller# (identify-caller (.getName (.getClass this#)) stack#)
                    function# (str (.getClassName caller#) "/" (.getMethodName caller#))
                    ~file-sym (.getFileName caller#)
                    ~line-sym (.getLineNumber caller#)]
                (binding [log/*ns* (if caller#
                                     (.getClassName caller#)
                                     (.getName this#))
                          log/*fn* function#
                          log/*file* ~file-sym
                          log/*line* ~line-sym]
                  ~(case signature
                     "-String"
                     `(let [[msg#] ~args-sym]
                        (log/log ~level [msg#]))

                     "-String-Object"
                     `(let [[fmt# o#] ~args-sym
                            ft# (MessageFormatter/format fmt# o#)]
                        (log/log ~level [(.getMessage ft#)] (.getThrowable ft#)))

                     "-String-Object-Object"
                     `(let [[fmt# o1# o2#] ~args-sym
                            ft# (MessageFormatter/format fmt# o1# o2#)]
                        (log/log ~level [(.getMessage ft#)] (.getThrowable ft#)))

                     "-String-Object<>"
                     `(let [[fmt# os#] ~args-sym
                            ft# (MessageFormatter/arrayFormat fmt# os#)]
                        (log/log ~level [(.getMessage ft#)] (.getThrowable ft#)))

                     "-String-Throwable"
                     `(let [[msg# e#] ~args-sym]
                        (log/log ~level [msg#] e#))))))))))

(define-methods "-error" :error)
(define-methods "-warn"  :warn)
(define-methods "-info"  :info)
(define-methods "-debug" :debug)
(define-methods "-trace" :trace)

(defn -log
  "-log method of LocationAwareLogger interface used by log4j-over-slf4j etc"
  [^SpectatorLoggerAdapter this ^Marker _marker fqcn level-const fmt arg-array e]
  (let [levels {LocationAwareLogger/ERROR_INT :error
                LocationAwareLogger/WARN_INT  :warn
                LocationAwareLogger/INFO_INT  :info
                LocationAwareLogger/DEBUG_INT :debug
                LocationAwareLogger/TRACE_INT :trace}
        level-keyword (levels level-const)
        stack (.getStackTrace (Thread/currentThread))
        ^StackTraceElement caller (identify-caller fqcn stack)
        message (.getMessage (MessageFormatter/arrayFormat fmt arg-array))]
    (binding [*ns* (.getName this)]
      (if caller ; nil when fqcn provided is incorrect (not present in call stack)
        (binding [log/*file* (.getFileName caller)
                  log/*line* (.getLineNumber caller)]
          (log/log level-keyword [message] e))
        (log/log level-keyword [message] e)))))

(defn -isErrorEnabled
  ([^SpectatorLoggerAdapter _this] true)
  ([^SpectatorLoggerAdapter _this _] true))

(defn -isWarnEnabled
  ([^SpectatorLoggerAdapter _this] true)
  ([^SpectatorLoggerAdapter _this _] true))

(defn -isInfoEnabled
  ([^SpectatorLoggerAdapter _this] true)
  ([^SpectatorLoggerAdapter _this _] true))

(defn -isDebugEnabled
  ([^SpectatorLoggerAdapter _this] true)
  ([^SpectatorLoggerAdapter _this _] true))

(defn -isTraceEnabled
  ([^SpectatorLoggerAdapter _this] true)
  ([^SpectatorLoggerAdapter _this _] true))


;;; Private

(defn- identify-caller ^StackTraceElement
  [fqcn stack]
  (letfn [(matches [^StackTraceElement el]
            (= 0 (st/index-of (str (.getClassName el) "$") (str fqcn "$"))))]
    (->> stack
         (drop-while (comp not matches))
         (drop-while matches)
         first)))
