;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns spectator.logger.console
  (:require [spectator.log :refer [ILogger]]
            [fluxus.stream :as s]
            [integrant.core :as ig]))

(declare print-log)

(defmethod ig/init-key :spectator.logger/console
  [_ {:keys [buffer-size]
      :or {buffer-size 100}}]
  #?(:cljs (enable-console-print!))
  (let [log-stream (s/stream (s/buffer buffer-size))]
    (s/consume print-log log-stream)
    (reify
      ILogger
      (record [_this log-entry]
        (s/put! log-stream log-entry)
        nil)
      (close! [_this]
        (s/close! log-stream)))))

(defmethod ig/halt-key! :spectator.logger/console [_ logger]
  (spectator.log/close! logger))


;;; Private

(defn- print-log
  [{:keys [ts function file line message error] :as entry}]
  (println (str ts) function (str "[" file ":" line "]")
           (pr-str message))
  (when error
    (println "  " :error (:type error))
    (println "  " (:message error))
    (when-let [d (:data error)]
      (println "  " (pr-str d)))
    (doseq [{:keys [function file line]} (:stack error)]
      (println "    " function (str "[" file ":" line "]")))))
