;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns spectator.log
  (:require [tempus.core :as t]
            [integrant.core :as ig]
            #?(:clj [clojure.main])
            #?(:cljs [utilis.js :as j])))

(defonce log-backend (atom nil))
(defonce log-level (atom nil))
(defonce log-service-name (atom nil))
(defonce log-service-version (atom nil))

(def levels {:fatal 0 :error 1 :warn 2 :info 3 :debug 4 :trace 5})

(defprotocol ILogger
  (record [logger log-entry])
  (close! [logger]))

(defmethod ig/init-key :spectator/log
  [_ {:keys [logger level service-name service-version]
      :or {level :info}}]
  (reset! log-backend logger)
  (reset! log-level level)
  (reset! log-service-name service-name)
  (reset! log-service-version service-version))

(defmethod ig/halt-key! :spectator/log [_ _]
  (reset! log-backend nil)
  (reset! log-level nil)
  (reset! log-service-name nil)
  (reset! log-service-version nil))

(declare parse-stack-frame parse-throwable)

(def pid #?(:clj (.pid (java.lang.ProcessHandle/current)) :cljs nil))

(defmacro log
  [level message & [error]]
  `(when (and @log-backend @log-level
              (<= (get levels ~level) (get levels @log-level)))
     (let [error# (when (instance? #?(:clj Throwable :cljs js/Error) ~error)
                    ~error)]
       (record
        @log-backend
        (merge
         {:ts (t/now)
          :level ~level
          :service {:name @log-service-name
                    :version @log-service-version}
          :ns ~(str *ns*)
          :message ~message}
         #?(:clj
            {:process {:pid ~pid
                       :thread (.getName (Thread/currentThread))}})
         #?(:clj
            (parse-stack-frame (-> (Throwable.) .getStackTrace first)))
         (when error#
           (parse-throwable error#)))))))

(defmacro trace
  [& args]
  `(log :trace ~@args))

(defmacro debug
  [& args]
  `(log :debug ~@args))

(defmacro info
  [& args]
  `(log :info ~@args))

(defmacro warn
  [& args]
  `(log :warn ~@args))

(defmacro error
  [& args]
  `(log :error ~@args))

(defmacro fatal
  [& args]
  `(log :fatal ~@args))

(defn parse-stack-frame
  [^StackTraceElement stack-frame]
  #?(:clj
     (let [class (.getClassName stack-frame)
           method (.getMethodName stack-frame)
           filename (.getFileName stack-frame)
           line (.getLineNumber stack-frame)]
       {:function (if (re-find #"^.*\.clj.?$" filename)
                    (clojure.main/demunge class)
                    (str class "/" method))
        :file filename
        :line line})))

#?(:clj
   (defn parse-throwable
     [^Throwable error]
     {:error (merge
              {:type (str (class error))
               :message (.getMessage error)
               :stack (->> (.getStackTrace error)
                           (map parse-stack-frame)
                           distinct)}
              (when (instance? clojure.lang.ExceptionInfo error)
                {:data (ex-data error)}))}))

#?(:cljs
   (defn parse-throwable
     [error]
     {:error {:type (j/get error :name)
              :message (j/get error :message)
              :file (j/get error :fileName)
              :line (j/get error :lineNumber)
              :stack (j/get error :stack)}}))
