;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns spectator.log
  (:require [spectator.logger :as l]
            #?(:clj [clj-commons.format.exceptions])
            [tempus.core]
            [clojure.string :as st])
  #?(:clj (:import [java.lang Throwable])))

#?(:cljs (set! *warn-on-infer* true))

(def levels {:fatal 0 :error 1 :warn 2 :info 3 :debug 4 :trace 5})

(defonce logger (atom nil))
(defonce service
  #?(:clj
     (let [[_ pn v] (->> (slurp "project.clj")
                         (re-find #"defproject (.*) \"(.*)\""))]
       (atom {:name pn :version v}))
     :cljs (atom nil)))
(defonce matchers (atom {"" 3}))
(defonce ns-level-cache (atom {}))

(declare set-level!)

(defn set-logger!
  [logger]
  (some-> spectator.log/logger deref deref l/close!)
  (reset! spectator.log/logger logger))

(defn set-service!
  [name version logger]
  (reset! spectator.log/service {:name name :version version})
  (set-logger! logger))

(defn set-level!
  [level]
  (swap! spectator.log/matchers assoc "" (get levels level)))

(defn reset-level!
  [level]
  (reset! ns-level-cache {})
  (reset! spectator.log/matchers {"" (get levels level)}))

(defn set-ns-level!
  [ns-prefix level]
  (reset! ns-level-cache {})
  (swap! spectator.log/matchers assoc ns-prefix (get levels level)))

#?(:clj (def pid (.pid (java.lang.ProcessHandle/current))))

(declare ns-level)

#?(:clj
   (defmacro log
     [level message & [error]]
     (let [ns (str *ns*)]
       `(when (and @logger
                   (<= (get levels ~level) (ns-level ~ns)))
          (let [entry# (merge
                        {:ts (tempus.core/now)
                         :level ~level
                         :service @spectator.log/service
                         :ns ~ns
                         :message ~message
                         :error ~error}
                        #?(:clj
                           (-> (clj-commons.format.exceptions/analyze-exception
                                (Throwable.) nil)
                               first
                               :stack-trace
                               first
                               ((fn [frame#]
                                  {:function (:name frame#)
                                   :file (:file frame#)
                                   :line (:line frame#)}))))
                        #?(:clj
                           {:process {:pid ~pid
                                      :thread (.getName (Thread/currentThread))}}))]
            (l/record @@logger entry#)
            nil)))))

#?(:clj
   (defmacro trace
     [& args]
     `(log :trace ~@args)))

#?(:clj
   (defmacro debug
     [& args]
     `(log :debug ~@args)))

#?(:clj
   (defmacro info
     [& args]
     `(log :info ~@args)))

#?(:clj
   (defmacro warn
     [& args]
     `(log :warn ~@args)))

#?(:clj
   (defmacro error
     [& args]
     `(log :error ~@args)))

#?(:clj
   (defmacro fatal
     [& args]
     `(log :fatal ~@args)))

(defn ns-level
  [ns]
  (let [ns (str ns)]
    (or (get @ns-level-cache ns)
        (let [level (->> @spectator.log/matchers
                         (filter (fn [[prefix _]]
                                   (st/starts-with? ns prefix)))
                         (sort-by (comp count first))
                         last last)]
          (swap! ns-level-cache assoc ns level)
          level))))
