;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns spectator.logger.text
  (:require [spectator.log :refer [service]]
            #?@(:clj [[clj-commons.format.exceptions :as ex]
                      [clj-commons.ansi :refer [compose]]]
                :cljs [[spectator.util.exception :as ex]])
            [spectator.logger :refer [ILogger]]
            [fluxus.flow :as f]
            [tempus.core :as t]
            #?@(:clj [[tempus.calendar :as tc]
                      [utilis.fs :as fs]])
            [clojure.string :as st]))

#?(:cljs (set! *warn-on-infer* true))

(declare render ->file)

(defn logger
  [{:keys [type buffer-size file-prefix]
    :or {buffer-size #?(:clj 1000 :cljs 100)}
    :as opts}]
  #?(:cljs (enable-console-print!))
  (when-not (or (= :console type)
                (and (= :file type) file-prefix))
    (throw (ex-info ":spectator.logger/text invalid-type" opts)))
  (delay
    (let [previous (volatile! nil)
          log #?(:clj (if (= :console type)
                        (comp println render last)
                        (let [path-sep (System/getProperty "file.separator")
                              path (st/split file-prefix (re-pattern path-sep))]
                          (when (> (count path) 1)
                            (fs/mkdir (str
                                       (when (st/starts-with? file-prefix path-sep)
                                         path-sep)
                                       (st/join path-sep (drop-last path)))
                                      :recursive true))
                          (partial ->file file-prefix)))
                 :cljs (comp println last))
          s (f/flow {:buffer [:sliding buffer-size]
                     :on-close (fn [_]
                                 (log [nil {:ts (t/now)
                                            :level :info
                                            :message [:spectator/logger type :stop @service]}]))
                     :on-error (fn [error]
                                 (log [nil {:level :error
                                            :message [:spectator.logger/error  @service]
                                            :error error}]))})]
      (log [nil {:ts (t/now)
                 :level :info
                 :message [:spectator/logger type :start @service]}])
      (f/consume log s)
      (reify ILogger
        (record [_ entry]
          (f/put! s [@previous entry])
          (vreset! previous entry))
        (close! [_]
          (f/close! s))))))

(defn render
  [{:keys [ts file line level message error]}]
  (let [render (fn [& args]
                 #?(:clj (compose
                          (into
                           [(case level
                              (:fatal :error) :bold.red
                              :warn :bold.bright-yellow
                              :debug :blue
                              :green)]
                           (interleave args (repeat " "))))
                    :cljs (st/join " " args)))
        line (str (render (str ts) (str level) (pr-str message))
                  (when (and file line)
                    (let [f (str "[" file ":" line "]")]
                      #?(:clj (compose [:faint.green f])
                         :cljs f))))]
    (cond-> line
      error
      (str "\n"
           #?(:clj
              (ex/format-exception error)
              :cljs
              (let [error (ex/parse error)]
                (str "  " :error (:type error) "\n"
                     "  " (:message error) "\n"
                     (when-let [d (:data error)]
                       (str "  " (pr-str d) "\n"))
                     (reduce (fn [s {:keys [function file line]}]
                               (str "    " function " [" file ":" line "]\n")) "" (:stack error)))))))))


;;; Private

#?(:clj
   (defn- ->file
     [file-prefix [previous entry]]
     (let [file (str file-prefix ".log")]
       (when (and previous (not (tc/same? :hour (:ts previous) (:ts entry))))
         (fs/cp file (str file-prefix "-"
                          (t/format "YYYY-MM-dd'T'HH'Z.log'" (:ts previous))))
         (fs/truncate file))
       (spit file (str (render entry) "\n") :append true))))
