;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns spectator.log
  (:refer-clojure :exclude [*ns* *file* *line*])
  #?(:cljs (:require-macros [spectator.log]))
  (:require [spectator.logger :as l]
            #?@(:clj [[clj-commons.format.exceptions]
                      [utilis.fs :as fs]])
            [tempus.core]
            [clojure.string :as st])
  #?(:clj (:import [java.lang Throwable])))

#?(:cljs (set! *warn-on-infer* true))

(def levels {:fatal 0 :error 1 :warn 2 :info 3 :debug 4 :trace 5})

(defonce logger (atom nil))
(defonce service
  #?(:clj
     (atom
      (cond
        (fs/file? "project.clj")
        (let [[_ pn v] (->> (slurp "project.clj")
                            (re-find #"defproject (.*) \"(.*)\""))]
          {:name pn :version v})

        (fs/file? "META-INF/MANIFEST.MF")
        (let [m (->> (slurp "META-INF/MANIFEST.MF")
                     st/split-lines
                     (map #(vec (map st/trim (st/split % #":"))))
                     (into {}))]
          {:name (str (get m "Leiningen-Project-GroupId") "/"
                      (get m "Leiningen-Project-ArtifactId"))
           :version (get m "Leiningen-Project-Version")})))
     :cljs (atom {})))

(defonce matchers (atom {"" 3}))
(defonce ns-level-cache (atom {}))

(declare set-level!)

(defn set-logger!
  [logger]
  (some-> spectator.log/logger deref deref l/close!)
  (reset! spectator.log/logger logger))

(defn set-service!
  [name version logger]
  (reset! spectator.log/service {:name name :version version})
  (set-logger! logger))

(defn set-level!
  [level]
  (reset! ns-level-cache {})
  (swap! spectator.log/matchers assoc "" (get levels level)))

(defn set-ns-level!
  [ns-prefix level]
  (reset! ns-level-cache {})
  (swap! spectator.log/matchers assoc (clojure.core/name ns-prefix) (get levels level)))

(declare ns-level)

(def ^:dynamic *ns* nil)
(def ^:dynamic *fn* nil)
(def ^:dynamic *file* nil)
(def ^:dynamic *line* nil)

(defn process
  []
  #?(:clj  (.pid (java.lang.ProcessHandle/current))
     :cljs nil))

(defn thread
  []
  #?(:clj  (.getName (Thread/currentThread))
     :cljs nil))

(defn location-cljs
  [env]
  (when (:ns env)
    (let [ns (-> env :ns :name str)]
      (merge
       {:ns ns
        :file (-> env :ns :meta :file)}
       (select-keys env [:line :column])
       (when-let [name (-> env :fn-scope first :name)]
         {:fn (str ns "/" name)})))))

#?(:clj
   (defn location-clj
     []
     (-> (clj-commons.format.exceptions/analyze-exception
          (Throwable.) nil)
         first
         :stack-trace
         second
         ((fn [frame#]
            {:fn (:name frame#)
             :file (str (when-let [p (:package frame#)] (str (st/replace p #"\." "/") "/")) (:file frame#))
             :line (:line frame#)})))))
#?(:clj
   (defmacro log
     [level message & [error]]
     (let [ns (or *ns* (str clojure.core/*ns*))]
       `(when (and @logger (<= (get levels ~level) (ns-level ~ns)))
          (l/record @@logger
                    (merge
                     {:ts (tempus.core/now)
                      :level ~level
                      :service @spectator.log/service
                      :ns ~ns
                      :message ~message
                      :error ~error
                      :process (process)
                      :thread (thread)}
                     ~(if (:ns &env)
                        (location-cljs &env)
                        `(location-clj))
                     (when *fn*
                       {:fn *fn*})
                     (when (and *file* *line*)
                       {:file *file*
                        :line *line*})))
          nil))))

#?(:clj
   (defmacro trace
     [& args]
     `(log :trace ~@args)))

#?(:clj
   (defmacro debug
     [& args]
     `(log :debug ~@args)))

#?(:clj
   (defmacro info
     [& args]
     `(log :info ~@args)))

#?(:clj
   (defmacro warn
     [& args]
     `(log :warn ~@args)))

#?(:clj
   (defmacro error
     [& args]
     `(log :error ~@args)))

#?(:clj
   (defmacro fatal
     [& args]
     `(log :fatal ~@args)))

(defn ns-level
  [ns]
  (let [ns (str ns)]
    (or (get @ns-level-cache ns)
        (let [level (->> @spectator.log/matchers
                         (filter (fn [[prefix _]]
                                   (st/starts-with? ns prefix)))
                         (sort-by (comp count first))
                         last last)]
          (swap! ns-level-cache assoc ns level)
          level))))
