;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns spectator.logger.text
  (:require [spectator.log :refer [service]]
            #?@(:clj [[clj-commons.format.exceptions :as ex]
                      [clj-commons.ansi :refer [compose]]
                      [tempus.calendar :as tc]
                      [utilis.fs :as fs]]
                :cljs [[utilis.js :as j]
                       ["ansicolor" :as ansi]])
            [spectator.logger :refer [ILogger]]
            [fluxus.flow :as f]
            [tempus.core :as t]
            [utilis.string :as ust]
            [clojure.string :as st]))

#?(:cljs (set! *warn-on-infer* true))

(declare render ->file)

(defn logger
  [{:keys [type buffer-size file-prefix]
    :or {buffer-size #?(:clj 1000 :cljs 100)}
    :as opts}]
  #?(:cljs (enable-console-print!))
  (when-not (or (= :console type)
                (and (= :file type) file-prefix))
    (throw (ex-info ":spectator.logger/text invalid-type" opts)))
  (delay
    (let [previous (volatile! nil)
          log #?(:clj (if (= :console type)
                        (fn [[_ entry]]
                          (locking Object
                            (apply println (render entry))))
                        (let [path-sep (System/getProperty "file.separator")
                              path (st/split file-prefix (re-pattern path-sep))]
                          (when (> (count path) 1)
                            (fs/mkdir (str
                                       (when (st/starts-with? file-prefix path-sep)
                                         path-sep)
                                       (st/join path-sep (drop-last path)))
                                      :recursive true))
                          (partial ->file file-prefix)))
                 :cljs (fn [[_ {:keys [level error] :as entry}]]
                         (apply
                          (case level
                            :fatal js/console.error
                            :error js/console.error
                            :warn  js/console.warn
                            :info  js/console.info
                            :debug js/console.debug
                            :trace js/console.debug)
                          (let [output (render entry)]
                            (if error
                              [(st/join " " (butlast output)) (last output)]
                              [(st/join " " output)])))))
          s (f/flow {:buffer [:sliding buffer-size]
                     :on-close (fn [_]
                                 (log [nil {:ts (t/now)
                                            :level :info
                                            :message [:spectator/logger type :stop @service]}]))
                     :on-error (fn [_f v error]
                                 (log [nil {:level :error
                                            :message [:spectator.logger/error @service v error]
                                            :error error}]))})]
      (log [nil {:ts (t/now)
                 :level :info
                 :message [:spectator/logger type :start @service]}])
      (f/consume log s)
      (reify ILogger
        (record [_ entry]
          (f/put! s [@previous entry])
          (vreset! previous entry))
        (close! [_]
          (f/close! s))))))

#?(:cljs
   (defn parse-ex
     [e]
     (cond->> [(let [new-err (js/Error. (j/get e :message))]
                 (cond->> (j/get e :stack)
                   (instance? cljs.core.ExceptionInfo e)
                   ((fn [stack]
                      (let [[error & frames] (st/split-lines stack)]
                        (->> frames
                             (drop-while #(not (or (st/starts-with? % "    at eval")
                                                   (st/starts-with? % "    at Object."))))
                             (map #(st/replace % #"Object\." ""))
                             (cons error)
                             (st/join "\n")))))
                   true (j/assoc! new-err :stack)))]
       (instance? cljs.core.ExceptionInfo e)
       (cons {:message (j/get e :message)
              :data (ex-data e)}))))

(defn render
  [{:keys [ts file line column level message error] :as opts}]
  (try
    (let [render (fn [& args]
                   #?(:clj (compose
                            (into
                             [(case level
                                (:fatal :error) :bold.red
                                :warn :bold.bright-yellow
                                :debug :blue
                                :trace :bold.blue
                                :green)]
                             (->> (interleave args (repeat " "))
                                  drop-last)))
                      :cljs ((case level
                               (:fatal :error) (comp ansi/bright ansi/red)
                               :warn (comp ansi/bright ansi/red)
                               :debug ansi/blue
                               :trace (comp ansi/bright ansi/blue)
                               ansi/green)
                             (st/join " " args))))
          render-loc (fn [v]
                       #?(:clj (compose [:faint.green v])
                          :cljs (when (not-empty v)
                                  (ansi/dim (ansi/green v)))))]
      (cond->
          [(render (str ts) (ust/format "%6s" level) (pr-str message))
           (render-loc
            (str
             (when-let [f (:fn opts)]
               (str "[" f "]"))
             (when file
               (str "[" file (when line (str ":" line)) (when column (str ":" column)) "]"))))]
        error
        #?(:clj  (conj (ex/format-exception error))
           :cljs (concat (parse-ex error)))))
    (catch #?(:clj Throwable :cljs :default) e
      (#?(:clj println :cljs js/console.error)
       (st/join " " [(str ts) (ust/format "%6s" :error) [:spectator.log/render (pr-str message)]])
       e)
      [::render-error])))


;;; Private

#?(:clj
   (defn- ->file
     [file-prefix [previous entry]]
     (let [file (str file-prefix ".log")]
       (when (and previous (not (tc/same? :hour (:ts previous) (:ts entry))))
         (fs/cp file (str file-prefix "-"
                          (t/format "YYYY-MM-dd'T'HH'Z.log'" (:ts previous))))
         (fs/truncate file))
       (spit file (str (render entry) "\n") :append true))))
