;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns structor-task.core
  (:require #?@(:bb [] :clj [[clj-commons.ansi :refer [compose]]])
            [babashka.fs :as fs]))

(defonce ^:dynamic *current-task* nil)
(defonce ^:dynamic *indent-level* -1)

(defmacro deftask
  [task-name config & fn-body]
  (let [[config fn-body] (if (map? config)
                           [config fn-body]
                           [nil (cons config fn-body)])
        fn-body (cond-> fn-body
                  (vector? (first fn-body))
                  list
                  (not-empty (:depends config))
                  (->> (map (fn [[args & body]]
                              `(~args
                                (doall
                                 (map #(apply % [])
                                      ~(->> (:depends config)
                                            (mapv (fn [dep#]
                                                    (if (symbol? dep#)
                                                      `(fn [] (~dep#))
                                                      `(fn [] ~dep#)))))))
                                ~@body)))))
        parse-arg (fn parse-arg [a]
                    (cond
                      (= '& a) nil
                      (map? a) (mapcat
                                (fn [[k v]]
                                  (cond
                                    (= :keys k)
                                    (mapcat (fn [k] [(keyword k) k]) v)

                                    (#{:as :or} k) nil

                                    (keyword? v)
                                    (into [v] (parse-arg k))

                                    (symbol? k)
                                    [(keyword k) k]

                                    :else (into [v] (parse-arg k))))
                                a)
                      (vector? a) [(vec (mapcat parse-arg a))]
                      :else [a]))]
    `(defn ~task-name
       ~@(map (fn [[args & body]]
                `(~args
                  (locking ~task-name
                    (let [start# (System/nanoTime)]
                      (binding [*current-task* ~(str (ns-name *ns*) "/" task-name)
                                *indent-level* (inc *indent-level*)]
                        (if ~(if (or (:outputs config) (:inputs config))
                               `(or (not (every? fs/exists? ~(:outputs config)))
                                    (seq
                                     (mapcat #(fs/modified-since % (vec (flatten ~(:inputs config))))
                                             ~(:outputs config))))
                               true)
                          (try
                            (log :start ~(vec (mapcat parse-arg args)))
                            ~@body
                            (doseq [o# ~(:outputs config)]
                              (when-not (fs/exists? o#)
                                (throw (ex-info "structor/task output missing"
                                                {:task ~(str task-name)
                                                 :missing o#}))))
                            (finally
                              (log :end :elapsed (format "%.3f ms" (/ (double (- (System/nanoTime) start#)) 1000000.0)))))
                          (log :no-op ~(if-let [msg (:no-op-msg config)] msg "inputs unmodified")))))))) fn-body))))

(defonce ^:private date-formatter
  (let [formatter (java.text.SimpleDateFormat. "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'")]
    (.setTimeZone formatter (java.util.TimeZone/getTimeZone "UTC"))
    formatter))

(defn log
  [& args]
  (locking Object
    (println
     (#?(:bb identity :clj compose)
      (#?@(:bb [str] :clj [vector :faint])
       (.format date-formatter (java.util.Date.))
       " " :build " "
       (apply str (repeat *indent-level* "  "))
       *current-task* " "
       (pr-str (vec args)))))))
