;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns structor.builder
  (:refer-clojure :exclude [read-string])
  (:require [structor.shadow-cljs :as shadow-cljs]
            [structor.clj-kondo]
            [structor.tailwind :as tailwind]
            [structor.tailwind-rn :as tailwind-rn]
            [structor.electron :as electron]
            [structor.available :refer [rn? npm? electron?]]
            [structor.sh :as sh]
            [utilis.fn :refer [fsafe]]
            [integrant.core :as ig]))

(declare watch stop clean)

(defmethod ig/init-key :structor.builder/watcher [_ {:keys [hooks] :as opts}]
  (watch opts))

(defmethod ig/halt-key! :structor.builder/watcher [_ watchers]
  (stop watchers))

(defn init
  []
  (println "npm init...")
  (sh/run ["sh" "-c" ["npm" "install"]])
  (sh/run ["npx" "browserslist@latest" "--update-db"])
  (println "done"))

(defn release
  ([] (release nil))
  ([{:keys [hooks] :as opts}]
   (println "clean...")
   (clean)
   (println "done")
   (when (npm?) (init))
   (let [init-result (when-let [h (:init hooks)]
                       (println "init hooks...")
                       (h opts)
                       (println "done"))
         {:keys [hooks
                 electron
                 tailwind
                 tailwind-rn
                 shadow-cljs]
          :as opts} (merge opts
                           (when (map? init-result)
                             init-result))]
     (when (rn?)
       (tailwind-rn/write-dummy-js))
     (println "shadow-cljs release...")
     (shadow-cljs/release shadow-cljs)
     (println "done")
     (println "tailwind release...")
     (tailwind/release
      (merge (when (rn?)
               {:input-file tailwind-rn/default-input-tailwind-css
                :output-file tailwind-rn/default-output-tailwind-css
                :minify false})
             tailwind))
     (println "done")
     (when (rn?)
       (println "tailwind-rn release...")
       (tailwind-rn/release tailwind-rn)
       (println "done")
       (println "rebuilding with tailwind-rn utilities...")
       (shadow-cljs/release shadow-cljs)
       (println "done"))
     (when (not (rn?))
       (println "uberjar...")
       (sh/run ["lein" "uberjar"])
       (println "done")
       (when-let [h (:uberjar hooks)]
         (println "uberjar hooks...")
         (h opts)
         (println "done")))
     (when (and electron (electron?))
       (println "electron release...")
       (electron/release)
       (println "done")))))

(defn watch
  ([] (watch nil))
  ([{:keys [hooks] :as opts}]
   (binding [sh/*trace* true]
     (println "clean...")
     (clean)
     (println "done")
     (when (npm?) (init))
     (println "clj-kondo init...")
     (ig/init-key :structor.clj-kondo/init {})
     (println "done")
     (let [init-result (when-let [h (:init hooks)]
                         (println "init hooks...")
                         (h opts)
                         (println "done"))
           {:keys [tailwind
                   tailwind-rn
                   shadow-cljs]} (merge opts
                                        (when (map? init-result)
                                          init-result))
           tailwind-watcher (do
                              (println "tailwind watch...")
                              (tailwind/watch
                               (merge (when (rn?)
                                        {:input-file tailwind-rn/default-input-tailwind-css
                                         :output-file tailwind-rn/default-output-tailwind-css})
                                      tailwind)))
           rn-watcher (when (rn?)
                        (tailwind-rn/watch tailwind-rn))]
       (println "shadow-cljs watch...")
       (merge {:shadow-cljs (shadow-cljs/watch shadow-cljs)
               :tailwind tailwind-watcher}
              (when rn-watcher {:tailwind-rn rn-watcher}))))))

(defn stop
  [watchers]
  (binding [sh/*trace* true]
    (shadow-cljs/stop (:shadow-cljs watchers))
    (tailwind/stop (:tailwind watchers))
    (when-let [tailwind-rn (:tailwind-rn watchers)]
      (tailwind-rn/stop tailwind-rn)))
  nil)

(defn clean
  []
  (shadow-cljs/clean)
  (if (rn?)
    (tailwind/clean {:output-file tailwind-rn/default-output-tailwind-css})
    (tailwind/clean))
  (tailwind-rn/clean)
  (electron/clean))
