;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns structor.android
  (:require [structor.artifact :as artifact]
            [structor.task :refer [deftask log]]
            [babashka.process :refer [shell]]
            [babashka.fs :as fs]
            [clojure.string :as st]))

(def home-dir (-> (shell {:out :string} "sh -c" "echo $HOME") :out st/trim))
(def adb (str home-dir "/Library/Android/sdk/platform-tools/adb"))

(def build-number
  (let [[major minor patch] (->> (st/split artifact/version #"\.")
                                 (map #(Long/parseLong %)))]
    (+ (* major 1000) (* minor 10) patch)))
(def apk-file (str artifact/group "_"
                   artifact/name "-"
                   artifact/version "_"
                   build-number ".apk"))

(defn devices
  []
  (->> (shell {:out :string} adb "devices") :out
       st/split-lines
       (map (comp second (partial re-find #"^(.*)\t(device|emulator)")))
       (remove nil?)))

(deftask usb-dev-enable
  []
  (let [forward (fn [device-id port]
                  (let [port (str "tcp:" port)]
                    (shell adb "-s" device-id "reverse" port port)))
        devices (devices)]
    (if (seq devices)
      (doseq [device-id devices]
        (log :enable device-id)
        (forward device-id 9630)
        (forward device-id 3449)
        (forward device-id 8081))
      (log :error "No Devices"))))

(deftask usb-dev-disable
  []
  (let [devices (devices)]
    (if (seq devices)
      (doseq [device-id devices]
        (log :disable device-id)
        (shell adb "-s" device-id "reverse" "--remove-all"))
      (log :error "No Devices found"))))

(deftask run
  []
  (shell "npx react-native run-android"))

(deftask metro
  []
  (shell "npx react-native start --reset-cache"))

(deftask dev-menu
  []
  (shell adb "shell" "input" "keyevent" "82"))

(deftask clean
  []
  (fs/delete-tree "app")
  (fs/create-dir "app"))

(deftask release
  [{:keys [build-type]
    :or {build-type :release}}]
  (let [build-type (name build-type)
        build-gradle "android/app/build.gradle"
        android-home (str home-dir "/Library/Android/sdk")]
    (-> (slurp build-gradle)
        (st/replace #"versionName\s+\".*\"" (str "versionName \"" artifact/version "\""))
        (st/replace #"versionCode\s+\d+" (str "versionCode " build-number))
        (->> (spit build-gradle)))
    (log :build apk-file)
    (shell {:dir "android"
            :extra-env {"JAVA_HOME" ""
                        "ANDROID_HOME" android-home}}
           (str "./gradlew assemble" (st/capitalize build-type)))
    (fs/copy (str "android/app/build/outputs/apk/" build-type "/app-" build-type ".apk")
             apk-file
             {:replace-existing true})))

(deftask install
  []
  (log :install apk-file)
  (shell adb "install" apk-file))
