;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns structor.electron
  (:refer-clojure :exclude [read-string])
  (:require [structor.task :refer [deftask]]
            [babashka.fs :as fs]
            [babashka.process :refer [shell]]
            [clojure.edn :refer [read-string]]
            [cheshire.core :as j]
            [clojure.java.io :as io]
            [clojure.string :as st]))

(declare copy-resource-files-to-build-directory
         copy-splash-files-to-build-directory
         copy-main-files-to-build-directory
         copy-config-files-to-build-directory)

(def electron-shell-resources
  ["package.json"
   "shadow-cljs.edn"
   "src/electron_shell/auto_updater.cljs"
   "src/electron_shell/config.cljs"
   "src/electron_shell/core.cljs"
   "src/electron_shell/debug.cljs"
   "src/electron_shell/download.cljs"
   "src/electron_shell/file_utils.cljs"
   "src/electron_shell/processes.cljs"
   "src/electron_shell/window.cljs"])

(def default-build-directory ".electron-shell")

(deftask clean
  []
  (fs/delete-tree default-build-directory)
  (fs/delete-tree "electron/dist"))

(defn bundle
  ([] (bundle default-build-directory))
  ([build-directory]
   (copy-resource-files-to-build-directory build-directory)
   (copy-splash-files-to-build-directory build-directory)
   (copy-main-files-to-build-directory build-directory)
   (copy-config-files-to-build-directory build-directory)))

(defn build
  ([] (build default-build-directory))
  ([build-directory]
   (shell {:dir build-directory} "npm install")
   (shell {:dir build-directory} "npx shadow-cljs release main")
   (shell {:dir build-directory} "npm run release")))

(defn copy-out-release-artifacts
  ([] (copy-out-release-artifacts default-build-directory))
  ([build-directory]
   (fs/create-dirs "electron/dist")
   (fs/copy-tree (format "%s/dist" build-directory) "electron/")))

(defn release
  ([] (release default-build-directory))
  ([build-directory]
   (clean build-directory)
   (bundle build-directory)
   (build build-directory)
   (copy-out-release-artifacts build-directory)))


;;; Private

(defn config-file
  []
  (let [config-file (io/file "electron/config.edn")]
    (when (.exists config-file)
      (read-string (slurp config-file)))))

(defn update-package-field
  ([package-json-string config-kp package-kp]
   (update-package-field package-json-string config-kp package-kp identity))
  ([package-json-string config-kp package-kp tx-value]
   (if-let [value (if (sequential? config-kp)
                    (get-in (config-file) config-kp)
                    (get (config-file) config-kp))]
     (let [value (tx-value value)]
       (cond-> (j/parse-string package-json-string true)
         (sequential? package-kp) (assoc-in package-kp value)
         (not (sequential? package-kp)) (assoc package-kp value)
         true (j/generate-string {:pretty true})))
     package-json-string)))

(defn maybe-set-package-field
  [package-json-string package-kp value]
  (when value
    (cond-> (j/parse-string package-json-string true)
      (sequential? package-kp) (assoc-in package-kp value)
      (not (sequential? package-kp)) (assoc package-kp value)
      true (j/generate-string {:pretty true}))))

(defn project-version
  []
  (try (-> (slurp "project.clj")
           read-string
           (nth 2))
       (catch Exception e
         nil)))

(defn copy-resource-files-to-build-directory
  [build-directory]
  (fs/create-dirs build-directory)
  (doseq [res electron-shell-resources]
    (when-let [directory (->> (-> res
                                  (st/split #"/")
                                  drop-last)
                              (st/join "/")
                              not-empty)]
      (fs/create-dirs (format "%s/%s" build-directory directory)))
    (let [contents (cond-> (->> (format "electron_shell/%s" res)
                                io/resource
                                slurp)
                     (= res "package.json") (-> (update-package-field :name "name")
                                                (update-package-field :author "author")
                                                (update-package-field :description "description")
                                                (update-package-field :app-id "appId")
                                                (update-package-field :artifact-name "artifactName")
                                                (update-package-field :app-icon ["build" "win" "icon"]
                                                                      (partial str "build/"))
                                                (update-package-field :app-icon ["build" "mac" "icon"]
                                                                      (partial str "build/"))
                                                (maybe-set-package-field "version" (project-version))))]
      (spit (format "%s/%s" build-directory res) contents)))
  (when-let [app-icon (:app-icon (config-file))]
    (fs/create-dirs (format "%s/build" build-directory))
    (fs/copy (format "electron/%s" app-icon)
             (format "%s/build" build-directory))))

(defn copy-config-files-to-build-directory
  [build-directory]
  (when-let [config (config-file)]
    (when (seq (:resources config))
      (fs/create-dirs (format "%s/extraResources" build-directory)))
    (doseq [resource (:resources config)]
      (let [resource (format "electron/%s" resource)
            file (io/file resource)]
        (if (.exists file)
          (fs/copy-tree resource (format "%s/extraResources" build-directory))
          (throw (ex-info (str "Resource file does not exist (" (str resource) ")")
                          {:resource resource})))))
    (->> (j/generate-string config {:pretty true})
         (spit (format "%s/config.json" build-directory)))))

(defn copy-splash-files-to-build-directory
  [build-directory]
  (let [splash-dir (io/file "electron/splash")]
    (when (and (.exists splash-dir)
               (.isDirectory splash-dir))
      (fs/copy-tree "electron/splash" build-directory))))

(defn copy-main-files-to-build-directory
  [build-directory]
  (let [main-dir (io/file "electron/main")]
    (when (and (.exists main-dir)
               (.isDirectory main-dir))
      (fs/copy-tree "electron/main" build-directory))))
