;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns structor.task
  (:require [babashka.fs :as fs]))

(defonce ^:private logging-lock (Object.))
(defonce ^:dynamic *structor-current-task* nil)
(defonce ^:dynamic *structor-indent-level* -1)

(defmacro deftask
  [task-name args & body]
  (let [[config body] (if (map? (first body))
                        [(first body) (rest body)]
                        [{} body])]
    `(defn ~task-name
       ~@(when (not-empty args)
           `[([] (~task-name {}))])
       (~args
        (let [start# (System/nanoTime)]
          (try
            (binding [*structor-current-task* ~(str (ns-name *ns*) "/" task-name)
                      *structor-indent-level* (inc *structor-indent-level*)]
              ~(when (not-empty (:depends config)) `(log :deps))
              ~@(map (fn [dep#] (if (symbol? dep#) `(~dep#) dep#)) (:depends config))
              (if ~(if (or (:outputs config) (:inputs config))
                     `(or (not (every? fs/exists? ~(:outputs config)))
                          (seq
                           (mapcat #(fs/modified-since % (vec (flatten ~(:inputs config))))
                                   ~(:outputs config))))
                     true)
                (try
                  (log :start ~args)
                  ~@body
                  (doseq [o# ~(:outputs config)]
                    (when-not (fs/exists? o#)
                      (throw (ex-info "output missing"
                                      {:task ~(str task-name)
                                       :missing o#}))))
                  (finally
                    (log :end :elapsed (format "%.3f ms" (/ (double (- (System/nanoTime) start#)) 1000000.0)))))
                (log :no-op ~(if-let [msg (:no-op-msg config)] msg "inputs unmodified"))))))))))

(def ^:private date-formatter (java.text.SimpleDateFormat. "yyyy-MM-dd' 'HH:mm:ss.SSS"))

(defn log
  [& args]
  (locking logging-lock
    (println (str (.format date-formatter (java.util.Date.)) ":"
                  (apply str (repeat *structor-indent-level* "  ")))
             *structor-current-task*
             (pr-str (vec args)))))
