;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns structor.docker
  (:require [structor-task.core :refer [deftask log]]
            [structor.artifact :as artifact]
            [babashka.process :refer [shell]]
            [babashka.fs :as fs]
            [clojure.string :as st]))

(def ^:dynamic *image-dir* "build/images")

(defn image-file
  [image]
  (-> image
      (st/replace (re-pattern "/") "_")
      (st/replace (re-pattern ":") "-")
      (str ".tar.gz")))

(defn image?
  [image version]
  (->> (shell {:out :string} "docker images")
       :out
       (re-find (re-pattern (str image "\\s+" version)))))

(deftask remove-image
  [image]
  (try
    (fs/delete-tree (str *image-dir* "/" (image-file image)))
    (if (->> (shell {:out :string :err :string} "docker images") :out
             (re-find (re-pattern (st/replace image #":" ".*"))))
      (do
        (log :removing image)
        (shell "docker rmi" image))
      (log :no-op image :does-not-exist))
    (catch Exception e
      (if (re-find #"Cannot connect to the Docker daemon at unix:///var/run/docker.sock"
                   (ex-message e))
        (log :error "Cannot connect to the docker daemon")
        (throw e)))))

(deftask clean
  []
  (remove-image artifact/docker-image))

(deftask build-image
  [image directory]
  (fs/create-dirs *image-dir*)
  (log :building image :in directory)
  (shell {:dir directory}
         "docker build -t" image ".")
  (let [export-file (str *image-dir* "/" (image-file image))]
    (log :exporting image :to export-file)
    (shell "sh -c" (str "docker save " image " | gzip > " export-file))))

(deftask image
  []
  (build-image artifact/docker-image "."))
