;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns structor.tailwind
  (:require [structor-task.core :refer [deftask]]
            [babashka.process :refer [shell process destroy-tree]]
            [babashka.fs :as fs]
            [clojure.string :as st]))

(def defaults
  {:input-css  "dev-resources/css/input.css"
   :input-js   "resources/public/js/compiled/main.js"
   :output-css "resources/public/css/main.css"})

(deftask clean
  [& {:keys [output-css]
      :or {output-css (:output-css defaults)}}]
  (fs/delete-if-exists output-css))

(deftask release
  {:depends [clean]
   :inputs (into [input-css "tailwind.config.js"]
                 (cond-> input-js (string? input-js) vector))
   :outputs [output-css]}
  [& {:keys [input-css output-css input-js minify]
      :or {input-css (:input-css defaults)
           input-js (fs/glob "resources/public/js/" "**/*.js")
           output-css (:output-css defaults)
           minify true}}]
  (shell {:extra-env {"NODE_ENV" "production"
                      "TAILWIND_MODE" "build"}}
         (st/join " " ["npx" "tailwindcss" "-i" input-css "-o" output-css
                       (when minify "--minify")]))
  (shell "touch" output-css))

(deftask watch
  [& {:keys [input-css output-css]
      :or {input-css (:input-css defaults)
           output-css (:output-css defaults)}}]
  (shell "npx" "tailwindcss" "-i" input-css "-o" output-css)
  (shell "touch" output-css)
  (process {:extra-env {"TAILWIND_MODE" "watch"}}
           "npx" "tailwindcss" "-i" input-css "-o" output-css "--watch"))

(deftask stop
  [watcher]
  (when watcher (destroy-tree watcher)))
