;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns structor.docker
  (:require [structor-task.core :refer [deftask log]]
            [structor.artifact :as artifact]
            [structor.cli :as cli]
            [babashka.process :refer [shell]]
            [babashka.fs :as fs]
            [clojure.string :as st]))

(def ^:dynamic *image-dir* "build/images")

(defn image-file
  [image & {:keys [arch]
            :or {arch (:arch cli/args)}}]
  (cli/force-arch arch)
  (-> image
      (st/replace (re-pattern "/") "_")
      (st/replace (re-pattern ":") "-")
      (str "-" arch ".tar.gz")))

(defn image?
  [image version]
  (->> (shell {:out :string} "docker images")
       :out
       (re-find (re-pattern (str image "\\s+" version)))))

(deftask remove-image
  [image & {:keys [os arch export]
            :or {os "linux"
                 arch (:arch cli/args)
                 export true}}]
  (cli/force-arch arch)
  (let [image (str image "-" arch)]
    (try
      (fs/delete-tree (str *image-dir* "/" (image-file image)))
      (if (->> (shell {:out :string :err :string} "docker images") :out
               (re-find (re-pattern (st/replace image #":" ".*"))))
        (do
          (log :removing image)
          (shell "docker rmi" image))
        (log :no-op image :does-not-exist))
      (catch Exception e
        (if (re-find #"Cannot connect to the Docker daemon at unix:///var/run/docker.sock"
                     (ex-message e))
          (log :error "Cannot connect to the docker daemon")
          (throw e))))))

(deftask clean
  []
  (remove-image artifact/docker-image))

(deftask build-image
  [image & {:keys [os arch dir export]
            :or {os "linux"
                 arch (:arch cli/args)
                 dir "."
                 export true}
            :as args}]
  (cli/force-arch arch)
  (let [image-file (apply image-file image args)
        image (str image "-" arch)]
    (fs/create-dirs *image-dir*)
    (log :building image :in dir)
    (shell {:dir dir}
           (str "docker buildx build --platform " (str os "/" arch)
                " -t " image " ."))
    (when export
      (let [export-file (str *image-dir* "/" image-file)]
        (log :exporting image :to export-file)
        (shell "sh -c" (str "docker save " image " | gzip > " export-file))))))

(deftask image
  [& args]
  (apply build-image artifact/docker-image args))

(deftask pull-image
  [image version & {:keys [os arch image-tag export]
                    :or {os "linux"
                         arch (:arch cli/args)
                         export false}
                    :as args}]
  (when-not (image? image (str version "-" arch))
    (log :pulling image version)
    (let [target-platform (str os "/" arch)]
      (shell (str "docker pull --platform " target-platform " "
                  image ":" version)))
    (when image-tag
      (log :tagging image version)
      (shell (str "docker image tag " image ":" version " " (str image-tag "-" arch)))
      (shell (str "docker rmi " image ":" version))))
  (when (and image-tag export)
    (let [image-file (apply image-file image-tag args)
          export-file (str *image-dir* "/" image-file)]
      (log :exporting (str image-tag "-" arch) :to export-file)
      (shell "sh -c" (str "docker save " (str image-tag "-" arch) " | gzip > " export-file)))))
