;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns structor.testing
  #?(:cljs (:require-macros [structor.testing]))
  (:require [structor-task.core :refer [deftask log]]
            [structor.process :refer [run kill]]
            [structor.namespace :refer [classpath namespaces test? file->ns]]
            [structor.npm :as npm]
            [clojure.test :as test]
            [shadow.cljs.cli :as shadow]
            #?@(:bb []
                :clj
                [[fs-watch.core :as fs]
                 [integrant.core :as ig]]
                :cljs
                [[cljs.core.async :as a]
                 [cljs.test :refer-macros [async] :include-macros true]])))

(declare watch-clj stop-clj watch-cljs stop-cljs)

#?(:bb nil
   :clj
   (defmethod ig/init-key :structor.testing/watcher-clj [_ {:keys [] :as opts}]
     (watch-clj opts)))

#?(:bb nil
   :clj
   (defmethod ig/halt-key! :structor.testing/watcher-clj [_ watchers]
     (stop-clj watchers)))

#?(:bb nil
   :clj
   (defmethod ig/init-key :structor.testing/watcher-cljs [_ {:keys [] :as opts}]
     (watch-cljs opts)))

#?(:bb nil
   :clj
   (defmethod ig/halt-key! :structor.testing/watcher-cljs [_ watchers]
     (stop-cljs watchers)))

#?(:clj
   (defmacro async
     {:style/indent 1}
     [done & body]
     (if (:ns &env)
       `(cljs.test/async ~done
          (a/go
            ~@body))
       `(let [~done (fn [])]
          ~@body))))

#?(:clj
   (defmacro sleep
     {:style/indent 1}
     [timeout]
     (if (:ns &env)
       `(a/<! (a/timeout ~timeout))
       `(java.lang.Thread/sleep ~timeout))))

#?(:clj
   (defn run-tests
     [& namespaces]
     (doseq [ns namespaces] (require ns))
     (apply test/run-tests namespaces)))

#?(:clj
   (defn run-all-tests
     []
     (apply run-tests (filter test? (namespaces)))))

(deftask test-clj
  []
  #?(:bb (run "lein with-profile repl test")
     :clj (apply run-tests (filter test? (namespaces)))))

(deftask test-cljs-deps
  []
  (binding [structor-task.core/*indent-level* 1]
    (npm/ensure
     ["shadow-cljs" (:version (shadow/get-shadow-cljs-info))]
     ["karma" "6.4.3"]
     ["karma-chrome-launcher" "3.2.0"]
     ["karma-cljs-test" "0.1.0"])))

(deftask test-cljs
  {:depends [test-cljs-deps]}
  []
  (run "npx shadow-cljs compile test")
  (run "npx karma start --single-run"))

(deftask test-all
  {:depends [test-clj test-cljs]}
  [])

(deftask watch-clj
  [& {}]
  (run-all-tests)
  (fs/watch (fn [{:keys [type path]}]
              (when (= :modify type)
                (let [ns (file->ns path)]
                  (if (test? ns)
                    (run-tests ns)
                    (try
                      (run-tests (symbol (str ns "-test")))
                      (catch Throwable _ nil))))))
            (->> (classpath)
                 (filter #(.exists %))
                 (map str))))

(deftask stop-clj
  [watcher]
  (fs/stop watcher))


(deftask watch-cljs
  {:depends [test-cljs-deps]}
  [& {}]
  [(run "npx shadow-cljs watch test" :wait false)
   (run "npx karma start" :wait false)])

(deftask stop-cljs
  [[shadow karma]]
  (kill shadow)
  (kill karma))
