;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns structor.shadow
  (:require [structor.npm :as npm]
            [structor.available :refer [rn?]]
            [structor-task.core :refer [deftask log]]
            #?@(:bb [[structor.process :refer [run]]]
                :clj [[shadow.cljs.devtools.server :as server]
                      [shadow.cljs.devtools.api :as shadow]])
            [babashka.fs :as fs]
            [shadow.cljs.cli :as shadow-cli]
            #?@(:bb []
                :clj [[integrant.core :as ig]])))

(defonce web-output-file "resources/public/js/compiled/main.js")
(defonce rn-output-file "app/index.js")

(declare watch stop)

#?(:bb nil
   :clj
   (defmethod ig/init-key :structor.shadow/watcher [_ {:keys [hooks] :as opts}]
     (watch opts)))

#?(:bb nil
   :clj
   (defmethod ig/halt-key! :structor.shadow/watcher [_ watchers]
     (stop watchers)))

(deftask shadow-deps
  []
  (binding [structor-task.core/*indent-level* 1]
    (npm/ensure
     ["shadow-cljs" (:version (shadow-cli/get-shadow-cljs-info))])))

(deftask clean
  []
  (doseq [path [".shadow-cljs" "resources/public/js/compiled"
                (when (rn?) "app")]]
    (fs/delete-tree path)))

(deftask release
  {:depends [npm/install npm/outdated shadow-deps clean]
   :inputs (concat
            (fs/glob "src" "**.clj*")
            (fs/glob "prod" "**.clj*")
            ["project.clj" "node_modules"])
   :outputs [(if (rn?) rn-output-file web-output-file)]}
  [& {:keys [build-id build-config]
      :or {build-id :prod}}]
  #?(:bb
     (run ["npx shadow-cljs release" (name build-id)])
     :clj
     (if build-config
       (shadow/with-runtime
         (shadow/release* build-config {}))
       (shadow/release build-id))))

#?(:bb nil
   :clj
   (deftask watch
     {:depends [npm/install npm/outdated shadow-deps]}
     [& {:keys [build-id build-config]
         :or {build-id :dev}}]
     (server/start!)
     (if build-config
       (shadow/watch* build-config {})
       (shadow/watch build-id))))

#?(:bb nil
   :clj
   (deftask stop
     [_watcher]
     (server/stop!)))
