;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns structor.android
  (:require [structor.artifact :as artifact]
            [structor-task.core :refer [deftask log]]
            [structor.process :as p]
            [crusta.core :as sh]
            [babashka.fs :as fs]
            [clojure.string :as st]))

(def home-dir (System/getProperty "user.home"))
(def adb (str home-dir "/Library/Android/sdk/platform-tools/adb"))
(def apksigner (str home-dir "/Library/Android/sdk/build-tools/34.0.0/apksigner"))

(def build-number
  (let [[major minor patch] (->> (st/split artifact/version #"\.")
                                 (map #(Long/parseLong %)))]
    (when (or (> patch 999)
              (> minor 999))
      (throw (ex-info "structor.android/build-number: versions too high" {:major major
                                                                          :minor minor
                                                                          :patch patch})))
    (+ (* major 1000000) (* minor 1000) patch)))

(def apk-file (str artifact/group "_"
                   artifact/name "-"
                   artifact/version "_"
                   build-number ".apk"))

(defn devices
  []
  (->> @(sh/run [adb "devices"])
       st/split-lines
       (map (comp second (partial re-find #"^(.*)\t(device|emulator)")))
       (remove nil?)))

(defn ->absolute
  [s]
  (st/replace s #"^~" (System/getProperty "user.home")))

(deftask usb-dev-enable
  [& ports]
  (let [forward (fn [device-id port]
                  (let [port (str "tcp:" port)]
                    (log :forward device-id port)
                    @(sh/run [adb "-s" device-id "reverse" port port])))
        devices (devices)]
    (if (seq devices)
      (doseq [device-id devices
              port (concat [9630 8081] ports)] ; shadow-cljs + metro + application ports
        (forward device-id port))
      (log :error "No Devices"))))

(deftask usb-dev-disable
  []
  (let [devices (devices)]
    (if (seq devices)
      (doseq [device-id devices]
        (log :disable device-id)
        (p/run [adb "-s" device-id "reverse" "--remove-all"]))
      (log :error "No Devices found"))))

(deftask run
  []
  (p/run "npx react-native run-android"))

(deftask metro
  []
  (p/run "npx react-native start --reset-cache"))

(deftask dev-menu
  []
  (p/run [adb "shell" "input" "keyevent" "82"]))

(deftask clean
  []
  (fs/delete-tree "app")
  (fs/create-dir "app"))

(deftask release
  [{:keys [build-type keystore keystore-props]
    :or {build-type :release}}]
  (let [build-type (name build-type)
        build-gradle "android/app/build.gradle"
        android-home (str home-dir "/Library/Android/sdk")]
    (-> (slurp build-gradle)
        (st/replace #"versionName\s+\".*\"" (str "versionName \"" artifact/version "\""))
        (st/replace #"versionCode\s+\d+" (str "versionCode " build-number))
        (->> (spit build-gradle)))
    (log :build apk-file)
    (let [cmd (str "./gradlew assemble" (st/capitalize build-type)
                   (when keystore (str " -PkeystoreFile='" (->absolute keystore) "'"))
                   (when keystore-props (str " -PkeystorePropertiesFile='" (->absolute keystore-props) "'")))]
      (log :cmd cmd)
      (p/run ["sh -c" [cmd]]
        {:directory "android"
         :environment {"JAVA_HOME" ""
                       "ANDROID_HOME" android-home}}))
    (fs/copy (str "android/app/build/outputs/apk/" build-type "/app-" build-type ".apk")
             apk-file
             {:replace-existing true})))

(deftask install
  []
  (log :install apk-file)
  (p/run [adb "install" apk-file]))

(deftask validate
  [{:keys [sha-256]}]
  (let [output @(sh/run [apksigner " verify --print-certs " apk-file])]
    (if (re-find (re-pattern (str "SHA-256 digest: " sha-256)) output)
      (log :validate :signature-valid output)
      (throw (ex-info "SHA-256 signature mismatch!" {:file apk-file
                                                     :sha-256 sha-256
                                                     :output output})))))
