;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns structor.tailwind
  (:require
   [babashka.fs :as fs]
   [fs-watch.core :as fsw]
   [structor-task.core :refer [deftask log]]
   [structor.process :refer [run]]
   [fluxus.flow :as f]))

(def defaults
  {:input-css  "dev-resources/css/input.css"
   :input-js   "resources/public/js/compiled/main.js"
   :output-css "resources/public/css/main.css"})

(deftask clean
  [& {:keys [output-css]
      :or {output-css (:output-css defaults)}}]
  (fs/delete-if-exists output-css))

(deftask release
  {:depends [clean]}
  [& {:keys [input-css output-css input-js minify]
      :or {input-css (:input-css defaults)
           output-css (:output-css defaults)
           minify true}}]
  (log :start :tailwind/release)
  (run ["tailwindcss" "-i" input-css "-o" output-css
        (when minify "--minify")]
    {:environment {"NODE_ENV" "production"
                   "TAILWIND_MODE" "build"}})
  (run ["touch" output-css])
  (log :end :tailwind/release))

(deftask watch
  [& {:keys [input-css output-css]
      :or {input-css (:input-css defaults)
           output-css (:output-css defaults)}}]
  (log :start :tailwind/watch)
  (let [run-f (f/flow {:buffer [:dropping 1]})]
    (f/consume  (fn [_]
                  (run ["tailwindcss" "-i" input-css "-o" output-css])) run-f)
    (f/put! run-f nil)
    {:watcher (fsw/watch (fn [change] (f/put! run-f change)) ["resources/public"])
     :runner run-f}))

(deftask stop
  [{:keys [watcher runner]}]
  (log :end :tailwind/watch)
  (fsw/stop watcher)
  (f/close! runner))
