;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns tempus.interval
  (:refer-clojure :exclude [into])
  (:require [tempus.core :as t]
            [tempus.duration :as td]
            [tempus.calendar :as tc]
            [utilis.inflections :refer [singular]]))

(defrecord Interval [start end bounds]
  Object
  (toString [_]
    (str "#tempus/interval [" (if (= :open (:left bounds)) ">= " "> ") start
         " " (if (= :open (:right bounds)) "<= " "< ") end "]")))

#?(:clj
   (defmethod print-method Interval [^Interval p ^java.io.Writer w]
     (.write w (.toString p))))

(defn interval
  [start end & {:keys [left-bound right-bound]
                :or {left-bound :closed
                     right-bound :open}}]
  (Interval. start end {:left left-bound :right right-bound}))

(defn into
  [time-unit i]
  (let [ms (- (t/into :long (:end i)) (t/into :long (:start i)))]
    (case time-unit
      :years (/ ms 3.154e+10)
      :weeks (/ ms 6.048e+8)
      :days (/ ms 8.64e+7)
      :hours (/ ms 3.6e+6)
      :minutes (/ ms 60000)
      :seconds (/ ms 1000)
      :milliseconds ms
      :edn {:start (:start i)
            :end (:end i)
            :bounds (:bounds i)})))

(defn from
  [type value]
  (case type
    :edn (Interval. (:start value) (:end value) (:bounds value))))

(defn seq-of
  ([time-unit interval]
   (let [duration ((case time-unit
                     :years td/years
                     :months td/months
                     :days td/days
                     :hours td/hours
                     :minutes td/minutes
                     :seconds td/seconds
                     :milliseconds td/milliseconds) 1)]
     (loop [values []
            current (tc/start-of (singular time-unit) (:start interval))]
       (if-not (t/>= current (:end interval))
         (recur
          (conj values current)
          (t/+ current duration))
         values))))
  ([time-unit start end]
   (seq-of time-unit (interval start end))))

(defn within?
  [interval timestamp]
  (let [{:keys [start end bounds]} interval]
    (and ((if (= :open (:left bounds)) t/>= t/>) timestamp start)
         ((if (= :open (:right bounds)) t/<= t/<) timestamp end))))
