;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns tempus.core
  (:refer-clojure :exclude [+ - < > <= >= time second format into
                            assoc update])
  (:require [utilis.js :as j])
  (:import [goog.date UtcDateTime Interval]
           [goog.i18n DateTimeFormat DateTimeParse]))

(declare into format)

(deftype DateTime [date-time meta-map]
  Object
  (toString [this]
    (format "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'" this))

  IHash
  (hash [_]
    (hash [:tempus/date-time date-time]))

  IEquiv
  (equals [_ other]
    (j/call date-time :equals (.-date-time other)))

  IComparable
  (-compare [this o]
    (if (number? o)
      (clojure.core/- (into :long this) o)
      (clojure.core/- (into :long this) (into :long o))))

  IWithMeta
  (-with-meta [_ meta-map]
    (DateTime. date-time meta-map nil))

  IMeta
  (-meta [_]
    meta-map)

  IPrintWithWriter
  (-pr-writer [this w _opts]
    (->> (str "#tempus/date-time " this)
         (write-all w)))

  )

(defn date-time
  ([year month day hour minute]
   (date-time year month day hour minute 0 0))
  ([year month day hour minute second]
   (date-time year month day hour minute second 0))
  ([year month day hour minute second millisecond]
   (DateTime. (UtcDateTime. year (dec month) day
                            hour minute second millisecond) nil)))

(defn into
  [type ts]
  (let [to-long #(.valueOf (.-date-time %))]
    (case type
      :long (to-long ts)
      :edn (to-long ts)
      :native (.-date-time ts)
      :js-date (js/Date. (to-long ts)))))

(defn from
  [type value]
  (let [from-long (fn [ms]
                    (DateTime. (let [t (UtcDateTime.)]
                                 (j/call t :setTime ms)
                                 t) nil))]
    (case type
      :long (from-long value)
      :edn (from-long value)
      :native (->> (DateTime. value nil) (into :long) (from :long))
      :js-date (from-long (j/call value :getTime)))))

(defn format
  [pattern ts]
  (let [formatter (DateTimeFormat. pattern)]
    (j/call formatter :format (.-date-time ts))))

(defn parse
  ([ts-str]
   (parse "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'" ts-str))
  ([pattern ts-str]
   (let [parser (DateTimeParse. pattern)
         ts (UtcDateTime.)]
     (j/call parser :parse ts-str ts)
     (DateTime. ts nil))))

(defn now
  []
  (DateTime. (UtcDateTime.) nil))

(defn year
  [t]
  (j/call (.-date-time t) :getUTCFullYear))

(defn month
  [t]
  (inc (j/call (.-date-time t) :getUTCMonth)))

(defn day-of-week
  [t]
  (j/call (.-date-time t) :getUTCDay))

(defn day
  [t]
  (j/call (.-date-time t) :getUTCDate))

(defn hour
  [t]
  (j/call (.-date-time t) :getUTCHours))

(defn minute
  [t]
  (j/call (.-date-time t) :getUTCMinutes))

(defn second
  [t]
  (j/call (.-date-time t) :getUTCSeconds))

(defn millisecond
  [t]
  (j/call (.-date-time t) :getUTCMilliseconds))

(defn update
  [t unit update-fn & params]
  (date-time
   (if (= :year unit) (apply update-fn (year t) params) (year t))
   (if (= :month unit) (apply update-fn (month t) params) (month t))
   (if (= :day unit) (apply update-fn (day t) params) (day t))
   (if (= :hour unit) (apply update-fn (hour t) params) (hour t))
   (if (= :minute unit) (apply update-fn (minute t) params) (minute t))
   (if (= :second unit) (apply update-fn (second t) params) (second t))
   (if (= :millisecond unit) (apply update-fn (millisecond t) params) (millisecond t))))

(defn assoc
  [t unit value]
  (update t unit (constantly value)))

(defn +
  [ts & durations]
  (reduce (fn [ts duration]
            (DateTime. (let [ts (j/call (.-date-time ts) :clone)
                             i (case (.-unit duration)
                                 :years (Interval. (.-value duration))
                                 :months (Interval. 0 (.-value duration))
                                 :days (Interval. 0 0 (.-value duration))
                                 :hours (Interval. 0 0 0 (.-value duration))
                                 :minutes (Interval. 0 0 0 0 (.-value duration))
                                 :seconds (Interval. 0 0 0 0 0 (.-value duration))
                                 :milliseconds (Interval. 0 0 0 0 0 0 (.-value duration)))]
                         (j/call ts :add i)
                         ts) nil)) ts durations))

(defn -
  [ts & durations]
  (reduce (fn [ts duration]
            (DateTime.
             (let [ts (j/call (.-date-time ts) :clone)
                   i (case (.-unit duration)
                       :years (Interval. (.-value duration))
                       :months (Interval. 0 (.-value duration))
                       :days (Interval. 0 0 (.-value duration))
                       :hours (Interval. 0 0 0 (.-value duration))
                       :minutes (Interval. 0 0 0 0 (.-value duration))
                       :seconds (Interval. 0 0 0 0 0 (.-value duration))
                       :milliseconds (Interval. 0 0 0 0 0 0 (.-value duration)))]
               (j/call ts :add (j/call i :getInverse))
               ts) nil)) ts durations))

(defn >
  [& times]
  (->> (partition 2 1 times)
       (every? (fn [[a b]]
                 (clojure.core/>
                  (j/call (.-date-time a) :valueOf)
                  (j/call (.-date-time b) :valueOf))))))

(defn <
  [& times]
  (->> (partition 2 1 times)
       (every? (fn [[a b]] (clojure.core/<
                           (j/call (.-date-time a) :valueOf)
                           (j/call (.-date-time b) :valueOf))))))

(defn <=
  [& times]
  (->> (partition 2 1 times)
       (every? (fn [[a b]] (or (= a b) (< a b))))))

(defn >=
  [& times]
  (->> (partition 2 1 times)
       (every? (fn [[a b]] (or (= a b) (> a b))))))
