;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns tempus.duration
  (:refer-clojure :exclude [into])
  (:require [utilis.inflections :refer [singular plural]])
  #?(:clj (:import [clojure.lang IObj IMeta])))

(deftype Duration [value unit meta-map]
  Object
  (toString [_]
    (str "#tempus/duration [" value " :" (cond-> (name unit) (clojure.core/= value 1) singular) "]"))
  #?(:cljs IHash)
  (#?(:clj hashCode :cljs -hash) [_]
    (hash [:tempus/duration value unit]))
  #?(:cljs IEquiv)
  (#?(:clj equals :cljs -equiv) [this other]
    (and (= (#?(:clj .value :cljs .-value) this)
            (#?(:clj .value :cljs .-value) ^Duration other))
         (= (#?(:clj .unit :cljs .-unit) this)
            (#?(:clj .unit :cljs .-unit) ^Duration other))))

  #?(:clj IObj :cljs IWithMeta)
  (#?(:clj withMeta :cljs -with-meta)
    [_ meta-map]
    (Duration. value unit meta-map))

  IMeta
  (#?(:clj meta :cljs -meta)
    [_]
    meta-map)

  #?@(:cljs
      [IPrintWithWriter
       (-pr-writer [this w _opts] (write-all w (.toString this)))]))

#?(:clj
   (defmethod print-method Duration [^Duration d ^java.io.Writer w]
     (.write w (.toString d))))

(defn into
  [type ^Duration d]
  (case type
    :edn [(#?(:clj .value :cljs .-value) d) (#?(:clj .unit :cljs .-unit) d)]))

(defn from
  [type [value unit]]
  (case type
    :edn (Duration. value (-> unit plural keyword) nil)))

(defn years
  [value]
  (Duration. value :years nil))

(defn months
  [value]
  (Duration. value :months nil))

(defn weeks
  [value]
  (Duration. value :weeks nil))

(defn days
  [value]
  (Duration. value :days nil))

(defn hours
  [value]
  (Duration. value :hours nil))

(defn minutes
  [value]
  (Duration. value :minutes nil))

(defn seconds
  [value]
  (Duration. value :seconds nil))

(defn milliseconds
  [value]
  (Duration. value :milliseconds nil))
