;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns tempus.core
  (:refer-clojure :exclude [+ - < > <= >= time second into format
                            assoc update min max])
  (:require [tempus.duration])
  (:import [tempus.duration Duration]
           [clojure.lang IObj IMeta]
           [java.time OffsetDateTime Clock ZoneOffset Instant]
           [java.time.temporal ChronoUnit]
           [java.time.format DateTimeFormatter]
           [java.lang Comparable]))

(declare into format)

(deftype DateTime [^OffsetDateTime date-time meta-map]

  Object
  (toString [this]
    (format "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'" this))
  (hashCode [_]
    (hash [:tempus/date-time date-time]))
  (equals [_this other]
    (and (instance? DateTime other)
         (= date-time (.date-time  ^DateTime other))))

  Comparable
  (compareTo [this o]
    (let [result (if (number? o)
                   (clojure.core/- (into :long this) o)
                   (clojure.core/- (into :long this) (into :long o)))]
      (cond
        (pos? result) 1
        (neg? result) -1
        :else 0)))

  IObj
  (withMeta [_ meta-map]
    (DateTime. date-time meta-map))

  IMeta
  (meta [_]
    meta-map))

(defmethod print-method DateTime
  [^DateTime ts ^java.io.Writer w]
  (.write w (str "#tempus/date-time \"" (.toString ts) "\"")))

(defmethod print-dup DateTime
  [^DateTime ts ^java.io.Writer w]
  (.write w (str "#tempus/date-time \"" (.toString ts) "\"")))

(defn date-time
  ([year month day hour minute]
   (date-time year month day hour minute 0 0))
  ([year month day hour minute second]
   (date-time year month day hour minute second 0))
  ([year month day hour minute second millisecond]
   (DateTime. (OffsetDateTime/of year month day
                                 hour minute second (* 1000000 millisecond)
                                 ZoneOffset/UTC)
              nil)))

(defn into
  [type ts]
  (let [to-long #(let [^OffsetDateTime dt (.date-time ^DateTime %)]
                   (-> dt (.toInstant) (.toEpochMilli)))]
    (case type
      :long (to-long ts)
      :edn (to-long ts)
      :native (.date-time ^DateTime ts))))

(defn from
  [type value]
  (let [from-long #(DateTime. (-> (Instant/ofEpochMilli %)
                                  (OffsetDateTime/ofInstant ZoneOffset/UTC))
                              nil)]
    (case type
      :long (from-long value)
      :edn (from-long value)
      :native (->> (DateTime. value nil) (into :long) (from :long)))))

(defn format
  [pattern ts]
  (.format ^OffsetDateTime (.date-time ^DateTime ts) (DateTimeFormatter/ofPattern pattern)))

(defn parse
  ([ts-str]
   (parse "yyyy-MM-dd'T'HH:mm:ss.SSSVV" ts-str))
  ([pattern ts-str]
   (let [formatter (DateTimeFormatter/ofPattern pattern)]
     (DateTime. (OffsetDateTime/parse ts-str formatter) nil))))

(defn now
  []
  (DateTime. (OffsetDateTime/now (Clock/systemUTC)) nil))

(defn year
  [t]
  (.getYear ^OffsetDateTime (.date-time ^DateTime t)))

(defn month
  [t]
  (.getMonthValue ^OffsetDateTime (.date-time ^DateTime t)))

(defn day-of-week
  "The days of the week are ordered from 0-6,
  with 0 and 6 being Sunday and Saturday, respectively."
  [t]
  (let [weekday (.getValue (.getDayOfWeek ^OffsetDateTime (.date-time ^DateTime t)))]
    (clojure.core/mod weekday 7)))

(defn day
  [t]
  (.getDayOfMonth ^OffsetDateTime (.date-time ^DateTime t)))

(defn hour
  [t]
  (.getHour ^OffsetDateTime (.date-time ^DateTime t)))

(defn minute
  [t]
  (.getMinute ^OffsetDateTime (.date-time ^DateTime t)))

(defn second
  [t]
  (.getSecond ^OffsetDateTime (.date-time ^DateTime t)))

(defn millisecond
  [t]
  (quot (.getNano ^OffsetDateTime (.date-time ^DateTime t)) 1000000))

(defn- native-unit ^ChronoUnit
  [unit]
  (case unit
    :years ChronoUnit/YEARS
    :months ChronoUnit/MONTHS
    :days ChronoUnit/DAYS
    :hours ChronoUnit/HOURS
    :minutes ChronoUnit/MINUTES
    :seconds ChronoUnit/SECONDS
    :milliseconds ChronoUnit/MILLIS))

(defn update
  [t unit update-fn & params]
  (date-time
   (if (= :year unit) (apply update-fn (year t) params) (year t))
   (if (= :month unit) (apply update-fn (month t) params) (month t))
   (if (= :day unit) (apply update-fn (day t) params) (day t))
   (if (= :hour unit) (apply update-fn (hour t) params) (hour t))
   (if (= :minute unit) (apply update-fn (minute t) params) (minute t))
   (if (= :second unit) (apply update-fn (second t) params) (second t))
   (if (= :millisecond unit) (apply update-fn (millisecond t) params) (millisecond t))))

(defn assoc
  [t unit value]
  (update t unit (constantly value)))

(defn +
  [ts & durations]
  (reduce (fn [ts ^Duration duration]
            (DateTime. (.plus ^OffsetDateTime (.date-time ^DateTime ts)
                              ^long (.value duration)
                              (native-unit (.unit duration))) nil)) ts durations))

(defn -
  [ts & durations]
  (reduce (fn [ts ^Duration duration]
            (DateTime. (.minus ^OffsetDateTime (.date-time ^DateTime ts)
                               ^long (.value duration)
                               (native-unit (.unit duration))) nil)) ts durations))

(defn >
  [& times]
  (->> (partition 2 1 times)
       (every? (fn [[a b]]
                 (.isAfter ^OffsetDateTime (.date-time ^DateTime a)
                           ^OffsetDateTime (.date-time ^DateTime b))))))

(defn <
  [& times]
  (->> (partition 2 1 times)
       (every? (fn [[a b]]
                 (.isBefore ^OffsetDateTime (.date-time ^DateTime a)
                            ^OffsetDateTime (.date-time ^DateTime b))))))

(defn <=
  [& times]
  (->> (partition 2 1 times)
       (every? (fn [[a b]] (or (= a b) (< a b))))))

(defn >=
  [& times]
  (->> (partition 2 1 times)
       (every? (fn [[a b]] (or (= a b) (> a b))))))

(defn min
  "Returns the least of times."
  ([t] t)
  ([t1 t2] (if (<= t1 t2) t1 t2))
  ([t1 t2 & ts] (reduce min (min t1 t2) ts)))

(defn max
  "Returns the greatest of times."
  ([t] t)
  ([t1 t2] (if (>= t1 t2) t1 t2))
  ([t1 t2 & ts] (reduce max (max t1 t2) ts)))

(def min-value (from :long 0))

(def max-value (date-time 2200 1 1 0 0))
