;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns tempus.core
  (:refer-clojure :exclude [+ - < > <= >= time second into format
                            assoc update min max])
  (:require [tempus.duration])
  (:import [tempus.duration Duration]
           [clojure.lang IObj IMeta]
           [java.time ZonedDateTime ZoneId Instant Clock]
           [java.time.temporal ChronoUnit]
           [java.time.format DateTimeFormatter]
           [java.lang Comparable]))

(declare into format)

(defonce utc-clock (Clock/systemUTC))
(def ^:dynamic *clock* utc-clock)

(deftype DateTime [^ZonedDateTime date-time meta-map]

  Object
  (toString [this]
    (format "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'" this))
  (hashCode [_]
    (hash [:tempus/date-time date-time]))
  (equals [_this other]
    (and (instance? DateTime other)
         (= date-time (.date-time  ^DateTime other))))

  Comparable
  (compareTo [this o]
    (clojure.core/- (into :long this) (into :long o)))

  IObj
  (withMeta [_ meta-map]
    (DateTime. date-time meta-map))

  IMeta
  (meta [_]
    meta-map))

(defmethod print-method DateTime
  [^DateTime ts ^java.io.Writer w]
  (.write w (str "#tempus/date-time \"" (.toString ts) "\"")))

(defmethod print-dup DateTime
  [^DateTime ts ^java.io.Writer w]
  (.write w (str "#tempus/date-time \"" (.toString ts) "\"")))

(defn date-time
  ([year month day hour minute]
   (date-time year month day hour minute 0 0))
  ([year month day hour minute second]
   (date-time year month day hour minute second 0))
  ([year month day hour minute second millisecond]
   (DateTime. (ZonedDateTime/of
               year month day
               hour minute second (* millisecond 1000000)
               (ZoneId/of "UTC"))
              nil)))

(defn into
  [type ^DateTime ts]
  (let [millis (.toEpochMilli
                ^Instant (.toInstant
                          ^ZonedDateTime (.date-time ts)))]
    (case type
      :long millis
      :edn millis
      :native (.date-time ^DateTime ts))))

(defn from
  [type value]
  (let [ts (DateTime. (ZonedDateTime/ofInstant
                       (Instant/ofEpochMilli value)
                       (ZoneId/of "UTC")) nil)]
    (case type
      :long ts
      :edn ts
      :native (if (instance? ZonedDateTime value)
                (DateTime. value nil)
                (throw (Exception. "Invalid temporal value"))))))

(defn- formatter
  [pattern]
  (-> (DateTimeFormatter/ofPattern pattern)
      (.withZone (ZoneId/of "UTC"))))

(defn format
  [pattern ^DateTime ts]
  (.format ^ZonedDateTime (.date-time ts) (formatter pattern)))

(defn parse
  ([ts-str]
   (parse "yyyy-MM-dd'T'HH:mm:ss.SSSVV" ts-str))
  ([pattern ts-str]
   (DateTime. (ZonedDateTime/parse ts-str (formatter pattern))
              nil)))

(defn now
  []
  (DateTime. (ZonedDateTime/ofInstant
              (Instant/now *clock*)
              (ZoneId/of "UTC")) nil))

(defn year
  [t]
  (.getYear ^ZonedDateTime (.date-time ^DateTime t)))

(defn month
  [t]
  (.getMonthValue ^ZonedDateTime (.date-time ^DateTime t)))

(defn day-of-week
  "The days of the week are ordered from 0-6,
  with 0 and 6 being Sunday and Saturday, respectively."
  [t]
  (let [weekday (.getValue (.getDayOfWeek ^ZonedDateTime (.date-time ^DateTime t)))]
    (clojure.core/mod weekday 7)))

(defn day
  [t]
  (.getDayOfMonth ^ZonedDateTime (.date-time ^DateTime t)))

(defn hour
  [t]
  (.getHour ^ZonedDateTime (.date-time ^DateTime t)))

(defn minute
  [t]
  (.getMinute ^ZonedDateTime (.date-time ^DateTime t)))

(defn second
  [t]
  (.getSecond ^ZonedDateTime (.date-time ^DateTime t)))

(defn millisecond
  [t]
  (quot (.getNano ^ZonedDateTime (.date-time ^DateTime t)) 1000000))

(defn update
  [t unit update-fn & params]
  (date-time
   (if (= :year unit) (apply update-fn (year t) params) (year t))
   (if (= :month unit) (apply update-fn (month t) params) (month t))
   (if (= :day unit) (apply update-fn (day t) params) (day t))
   (if (= :hour unit) (apply update-fn (hour t) params) (hour t))
   (if (= :minute unit) (apply update-fn (minute t) params) (minute t))
   (if (= :second unit) (apply update-fn (second t) params) (second t))
   (if (= :millisecond unit) (apply update-fn (millisecond t) params) (millisecond t))))

(defn assoc
  [t unit value]
  (update t unit (constantly value)))

(defn- native-unit ^ChronoUnit
  [unit]
  (case unit
    :years ChronoUnit/YEARS
    :months ChronoUnit/MONTHS
    :days ChronoUnit/DAYS
    :hours ChronoUnit/HOURS
    :minutes ChronoUnit/MINUTES
    :seconds ChronoUnit/SECONDS
    :milliseconds ChronoUnit/MILLIS))

(defn +
  [ts & durations]
  (reduce (fn [ts ^Duration duration]
            (DateTime. (.plus ^ZonedDateTime (.date-time ^DateTime ts)
                              ^long (.value duration)
                              (native-unit (.unit duration)))
                       nil)) ts durations))

(defn -
  [ts & durations]
  (reduce (fn [ts ^Duration duration]
            (DateTime. (.minus ^ZonedDateTime (.date-time ^DateTime ts)
                               ^long (.value duration)
                               (native-unit (.unit duration)))
                       nil)) ts durations))

(defn <
  [& times]
  (->> (partition 2 1 times)
       (every? (fn [[a b]] (clojure.core/< (compare a b) 0)))))

(defn >
  [& times]
  (->> (partition 2 1 times)
       (every? (fn [[a b]] (clojure.core/> (compare a b) 0)))))

(defn <=
  [& times]
  (->> (partition 2 1 times)
       (every? (fn [[a b]] (clojure.core/<= (compare a b) 0)))))

(defn >=
  [& times]
  (->> (partition 2 1 times)
       (every? (fn [[a b]] (clojure.core/>= (compare a b) 0)))))

(defn min
  "Returns the least of times."
  ([t] t)
  ([t1 t2] (if (<= t1 t2) t1 t2))
  ([t1 t2 & ts] (reduce min (min t1 t2) ts)))

(defn max
  "Returns the greatest of times."
  ([t] t)
  ([t1 t2] (if (>= t1 t2) t1 t2))
  ([t1 t2 & ts] (reduce max (max t1 t2) ts)))

(def min-value (from :long 0))

(def max-value (date-time 2200 1 1 0 0))
