;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns tempus.core
  (:refer-clojure :exclude [+ - < > <= >= time second format into
                            assoc update min max])
  (:require [tempus.duration :refer [Duration]]
            [utilis.js :as j])
  (:import [goog.date UtcDateTime Interval]
           [goog.i18n DateTimeFormat DateTimeParse]))

(set! *warn-on-infer* true)

(declare into format)

(deftype DateTime [date-time meta-map]
  Object
  (toString [this]
    (format "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'" this))

  IHash
  (-hash [_]
    (hash [:tempus/date-time (.valueOf ^UtcDateTime date-time)]))

  IEquiv
  (-equiv [this ^DateTime other]
    (boolean
     (and (instance? DateTime other)
          (= (into :long this)
             (into :long other)))))

  IComparable
  (-compare [this o]
    (clojure.core/- (into :long this) (into :long o)))

  IWithMeta
  (-with-meta [_ meta-map]
    (DateTime. date-time meta-map))

  IMeta
  (-meta [_]
    meta-map)

  IPrintWithWriter
  (-pr-writer [this w _opts]
    (->> (str "#tempus/date-time " this)
         (write-all w))))

(defn date-time
  ([year month day hour minute]
   (date-time year month day hour minute 0 0))
  ([year month day hour minute second]
   (date-time year month day hour minute second 0))
  ([year month day hour minute second millisecond]
   (DateTime. (UtcDateTime. year (dec month) day
                            hour minute second millisecond) nil)))

(defn into
  [type ^DateTime ts]
  (let [to-long #(.valueOf (.-date-time ^DateTime %))]
    (case type
      :long (to-long ts)
      :edn (to-long ts)
      :native (.-date-time ts)
      :js-date (js/Date. (to-long ts)))))

(defn from
  ^DateTime
  [type value]
  (let [from-long (fn [ms]
                    (DateTime. (let [t (UtcDateTime.)]
                                 (j/call t :setTime ms)
                                 t) nil))]
    (case type
      :long (from-long value)
      :edn (from-long value)
      :native (->> (DateTime. value nil) (into :long) (from :long))
      :js-date (from-long (j/call value :getTime)))))

(defn format
  [pattern ^DateTime ts]
  (let [formatter (DateTimeFormat. pattern)]
    (j/call formatter :format (.-date-time ts))))

(defn parse
  ^DateTime
  ([ts-str]
   (parse "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'" ts-str))
  ([pattern ts-str]
   (let [parser (DateTimeParse. pattern)
         ts (UtcDateTime.)]
     (j/call parser :parse ts-str ts)
     (DateTime. ts nil))))

(defn now
  ^DateTime
  []
  (DateTime. (UtcDateTime.) nil))

(defn year
  [^DateTime t]
  (j/call (.-date-time t) :getUTCFullYear))

(defn month
  [^DateTime t]
  (inc (j/call (.-date-time t) :getUTCMonth)))

(defn day-of-week
  [^DateTime t]
  (j/call (.-date-time t) :getUTCDay))

(defn day
  [^DateTime t]
  (j/call (.-date-time t) :getUTCDate))

(defn hour
  [^DateTime t]
  (j/call (.-date-time t) :getUTCHours))

(defn minute
  [^DateTime t]
  (j/call (.-date-time t) :getUTCMinutes))

(defn second
  [^DateTime t]
  (j/call (.-date-time t) :getUTCSeconds))

(defn millisecond
  [^DateTime t]
  (j/call (.-date-time t) :getUTCMilliseconds))

(defn update
  [^DateTime t unit update-fn & params]
  (date-time
   (if (= :year unit) (apply update-fn (year t) params) (year t))
   (if (= :month unit) (apply update-fn (month t) params) (month t))
   (if (= :day unit) (apply update-fn (day t) params) (day t))
   (if (= :hour unit) (apply update-fn (hour t) params) (hour t))
   (if (= :minute unit) (apply update-fn (minute t) params) (minute t))
   (if (= :second unit) (apply update-fn (second t) params) (second t))
   (if (= :millisecond unit) (apply update-fn (millisecond t) params) (millisecond t))))

(defn assoc
  [^DateTime t unit value]
  (update t unit (constantly value)))

(defn +
  [^DateTime ts & durations]
  (reduce (fn [^DateTime ts ^Duration duration]
            (if (= :milliseconds (.-unit duration))
              (DateTime. (UtcDateTime/fromTimestamp
                          (clojure.core/+
                           (j/call (.-date-time ts) :valueOf)
                           (.-value duration))) nil)
              (DateTime. (let [ts (j/call (.-date-time ts) :clone)
                               i (case (.-unit duration)
                                   :years (Interval. (.-value duration))
                                   :months (Interval. 0 (.-value duration))
                                   :days (Interval. 0 0 (.-value duration))
                                   :hours (Interval. 0 0 0 (.-value duration))
                                   :minutes (Interval. 0 0 0 0 (.-value duration))
                                   :seconds (Interval. 0 0 0 0 0 (.-value duration)))]
                           (j/call ts :add i)
                           ts) nil))) ts durations))

(defn -
  [^DateTime ts & durations]
  (reduce (fn [^DateTime ts ^Duration duration]
            (if (= :milliseconds (.-unit duration))
              (DateTime. (UtcDateTime/fromTimestamp
                          (clojure.core/-
                           (j/call (.-date-time ts) :valueOf)
                           (.-value duration))) nil)
              (DateTime.
               (let [ts (j/call (.-date-time ts) :clone)
                     i (case (.-unit duration)
                         :years (Interval. (.-value duration))
                         :months (Interval. 0 (.-value duration))
                         :days (Interval. 0 0 (.-value duration))
                         :hours (Interval. 0 0 0 (.-value duration))
                         :minutes (Interval. 0 0 0 0 (.-value duration))
                         :seconds (Interval. 0 0 0 0 0 (.-value duration)))]
                 (j/call ts :add (j/call i :getInverse))
                 ts) nil))) ts durations))

(defn >
  [& times]
  (->> (partition 2 1 times)
       (every? (fn [[^DateTime a ^DateTime b]]
                 (clojure.core/>
                  (j/call (.-date-time a) :valueOf)
                  (j/call (.-date-time b) :valueOf))))))

(defn <
  [& times]
  (->> (partition 2 1 times)
       (every? (fn [[^DateTime a ^DateTime b]]
                 (clojure.core/<
                  (j/call (.-date-time a) :valueOf)
                  (j/call (.-date-time b) :valueOf))))))

(defn <=
  [& times]
  (->> (partition 2 1 times)
       (every? (fn [[^DateTime a ^DateTime b]]
                 (or (= a b) (< a b))))))

(defn >=
  [& times]
  (->> (partition 2 1 times)
       (every? (fn [[^DateTime a ^DateTime b]]
                 (or (= a b) (> a b))))))

(defn min
  "Returns the least of times."
  ([t] t)
  ([t1 t2] (if (<= t1 t2) t1 t2))
  ([t1 t2 & ts] (reduce min (min t1 t2) ts)))

(defn max
  "Returns the greatest of times."
  ([t] t)
  ([t1 t2] (if (>= t1 t2) t1 t2))
  ([t1 t2 & ts] (reduce max (max t1 t2) ts)))

(def min-value (from :long 0))

(def max-value (date-time 2200 1 1 0 0))
