;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns tempus.interval
  (:refer-clojure :exclude [into])
  (:require [tempus.core :as t]
            [tempus.duration :as td]
            [tempus.calendar :as tc]
            [utilis.inflections :refer [singular]])
  #?(:clj (:import [clojure.lang IObj IMeta])))

#?(:cljs (set! *warn-on-infer* true))

(deftype Interval [start end bounds meta-map]
  Object
  (toString [_]
    (str "#tempus/interval [" (if (= :open (:left bounds)) ">= " "> ") start
         " " (if (= :open (:right bounds)) "<= " "< ") end "]"))
  #?(:cljs IHash)
  (#?(:clj hashCode :cljs -hash) [_]
    (hash [:tempus/interval start end bounds]))
  #?(:cljs IEquiv)
  (#?(:clj equals :cljs -equiv) [this other]
    (and (= (#?(:clj .start :cljs .-start) this)
            (#?(:clj .start :cljs .-start) ^Interval other))
         (= (#?(:clj .end :cljs .-end) this)
            (#?(:clj .end :cljs .-end) ^Interval other))
         (= (#?(:clj .bounds :cljs .-bounds) this)
            (#?(:clj .bounds :cljs .-bounds) ^Interval other))))

  #?(:clj IObj :cljs IWithMeta)
  (#?(:clj withMeta :cljs -with-meta)
    [_ meta-map]
    (Interval. start end bounds meta-map))

  IMeta
  (#?(:clj meta :cljs -meta)
    [_]
    meta-map)

  #?@(:cljs
      [IPrintWithWriter
       (-pr-writer [this w _opts] (write-all w (.toString this)))]))

#?(:clj
   (defmethod print-method Interval [^Interval i ^java.io.Writer w]
     (.write w (.toString i))))

(defn interval
  [start end & {:keys [left-bound right-bound]
                :or {left-bound :closed
                     right-bound :open}}]
  (Interval. start end {:left left-bound :right right-bound} nil))

(defn start
  [^Interval interval]
  (#?(:clj .start :cljs .-start) interval))

(defn end
  [^Interval interval]
  (#?(:clj .end :cljs .-end) interval))

(defn into
  [time-unit ^Interval i]
  (let [ms (- (t/into :long (#?(:clj .end :cljs .-end) i))
              (t/into :long (#?(:clj .start :cljs .-start) i)))]
    (case time-unit
      :years (/ ms 3.154e+10)
      :weeks (/ ms 6.048e+8)
      :days (/ ms 8.64e+7)
      :hours (/ ms 3.6e+6)
      :minutes (/ ms 60000)
      :seconds (/ ms 1000)
      :milliseconds ms
      :edn {:start (#?(:clj .start :cljs .-start) i)
            :end (#?(:clj .end :cljs .-end) i)
            :bounds (#?(:clj .bounds :cljs .-bounds) i)})))

(defn from
  [type value]
  (case type
    :edn (Interval. (:start value) (:end value) (:bounds value) nil)))

(defn seq-of
  ([time-unit ^Interval interval]
   (let [duration ((case time-unit
                     :years td/years
                     :months td/months
                     :days td/days
                     :hours td/hours
                     :minutes td/minutes
                     :seconds td/seconds
                     :milliseconds td/milliseconds) 1)
         start (#?(:clj .start :cljs .-start) interval)
         end (#?(:clj .end :cljs .-end) interval)
         bounds (#?(:clj .bounds :cljs .-bounds) interval)
         comparator (if (= :open (:right bounds)) t/<= t/<)]
     (loop [values []
            current (tc/start-of (singular time-unit) start)]
       (if (comparator current end)
         (recur
          (conj values current)
          (t/+ current duration))
         values))))
  ([time-unit start end]
   (seq-of time-unit (interval start end))))

(defn within?
  [^Interval interval timestamp]
  (let [start (#?(:clj .start :cljs .-start) interval)
        end (#?(:clj .end :cljs .-end) interval)
        bounds (#?(:clj .bounds :cljs .-bounds) interval)]
    (and ((if (= :open (:left bounds)) t/>= t/>) timestamp start)
         ((if (= :open (:right bounds)) t/<= t/<) timestamp end))))
