;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns tempus.calendar
  (:require [tempus.duration :as td]
            [tempus.core :as t]))

(defn same?
  [time-unit ts1 ts2]
  (case time-unit
    :millisecond (= ts1 ts2)
    :second (and (= (t/second ts1) (t/second ts2))
                 (= (t/minute ts1) (t/minute ts2))
                 (= (t/hour ts1) (t/hour ts2))
                 (= (t/day ts1) (t/day ts2))
                 (= (t/month ts1) (t/month ts2))
                 (= (t/year ts1) (t/year ts2)))
    :minute (and (= (t/minute ts1) (t/minute ts2))
                 (= (t/hour ts1) (t/hour ts2))
                 (= (t/day ts1) (t/day ts2))
                 (= (t/month ts1) (t/month ts2))
                 (= (t/year ts1) (t/year ts2)))
    :hour (and (= (t/hour ts1) (t/hour ts2))
               (= (t/day ts1) (t/day ts2))
               (= (t/month ts1) (t/month ts2))
               (= (t/year ts1) (t/year ts2)))
    :day (and (= (t/day ts1) (t/day ts2))
              (= (t/month ts1) (t/month ts2))
              (= (t/year ts1) (t/year ts2)))
    :month (and (= (t/month ts1) (t/month ts2))
                (= (t/year ts1) (t/year ts2)))
    :year (= (t/year ts1) (t/year ts2))))

(defn start-of
  [time-unit ts & {:keys [week-start] :or {week-start :sunday}}]
  (case time-unit
    :second
    (t/date-time (t/year ts) (t/month ts) (t/day ts)
                 (t/hour ts) (t/minute ts) (t/second ts) 0)
    :minute
    (t/date-time (t/year ts) (t/month ts) (t/day ts)
                 (t/hour ts) (t/minute ts) 0 0)
    :hour
    (t/date-time (t/year ts) (t/month ts) (t/day ts)
                 (t/hour ts) 0 0 0)
    :day
    (t/date-time (t/year ts) (t/month ts) (t/day ts)
                 0 0 0 0)
    :week
    (let [day-delta (cond-> (t/day-of-week ts)
                      (= week-start :monday) dec)]
      (t/- (t/date-time (t/year ts) (t/month ts) (t/day ts) 0 0 0 0)
           (td/days day-delta)))
    :month
    (t/date-time (t/year ts) (t/month ts) 1
                 0 0 0 0)
    :year
    (t/date-time (t/year ts) 1 1
                 0 0 0 0)))

(defn end-of
  [time-unit ts & {:keys [week-start] :or {week-start :sunday}}]
  (case time-unit
    :second
    (t/date-time (t/year ts) (t/month ts) (t/day ts)
                 (t/hour ts) (t/minute ts) (t/second ts) 999)
    :minute
    (t/date-time (t/year ts) (t/month ts) (t/day ts)
                 (t/hour ts) (t/minute ts) 59 999)
    :hour
    (t/date-time (t/year ts) (t/month ts) (t/day ts)
                 (t/hour ts) 59 59 999)
    :day
    (t/date-time (t/year ts) (t/month ts) (t/day ts)
                 23 59 59 999)
    :week
    (let [day-delta (cond-> (- 6 (t/day-of-week ts))
                      (= week-start :monday) inc)]
      (t/+ (t/date-time (t/year ts) (t/month ts) (t/day ts)
                        23 59 59 999)
           (td/days day-delta)))
    :month
    (t/- (if (= (inc (t/month ts)) 13)
           (t/date-time (inc (t/year ts)) 1 1
                        0 0 0)
           (t/date-time (t/year ts) (inc (t/month ts)) 1
                        0 0 0))
         (td/milliseconds 1))
    :year
    (t/date-time (t/year ts) 12 31
                 23 59 59 999)))
