;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns tempus.series
  (:require [tempus.core :as t]
            [tempus.duration :as td #?@(:cljs [:refer [Duration]])]
            #?(:cljs [utilis.js :as j]))
  #?(:clj (:import [tempus.duration Duration])))

(defn periodic
  [start duration]
  (iterate (fn [ts] (t/+ ts duration)) start))

(defn exponential
  [start & {:keys [factor first-step max-step]
            :or {factor 2
                 first-step (td/seconds 1)}
            :as opts}]
  (when (and max-step (not= #?(:clj (.unit ^Duration first-step)
                               :cljs (j/get first-step :unit))
                            #?(:clj (.unit ^Duration max-step)
                               :cljs (j/get max-step :unit))))
    (throw (ex-info "Units of first-step and max-step must be the same"
                    opts)))
  (let [max-step (if max-step
                   #?(:clj (.value ^Duration max-step)
                      :cljs (j/get max-step :value))
                   #?(:clj  Long/MAX_VALUE
                      :cljs js/Number.MAX_SAFE_INTEGER))]
    (->> [start first-step]
         (iterate (fn [[ts ^Duration step]]
                    [(t/+ ts step)
                     (let [next-step (* factor
                                        #?(:clj (.value step)
                                           :cljs (j/get step :value)))]
                       (Duration. (min next-step max-step)
                                  #?(:clj (.unit step)
                                     :cljs (j/get step :unit))
                                  #?(:clj (.meta-map step)
                                     :cljs (j/get step :meta-map))))]))
         (map first))))
