;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns utilis.hash
  (:require [clojure.java.io :as io])
  (:import [java.security MessageDigest DigestInputStream]
           [java.io File InputStream]))

(declare compute)

(defn sha256
  [v]
  (->> (compute "sha256" v)
       (map #(.substring
              (Integer/toString
               (+ (bit-and % 0xff) 0x100) 16) 1))
       (apply str)))

(defn md5
  [v]
  (->> (BigInteger. 1 ^bytes (compute "MD5" v))
       (format "%032x")))


;;; Private

(defn- compute
  [algorithm v]
  (let [digest (MessageDigest/getInstance algorithm)]
    (cond
      (instance? InputStream v)
      (with-open [dis (DigestInputStream. v digest)]
        (while (pos? (.read dis)))
        (.digest digest))

      (instance? File v)
      (with-open [fis (io/input-stream v)]
        (compute algorithm fis))

      (bytes? v)
      (.digest digest v)

      (string? v)
      (compute algorithm (.getBytes ^String v)))))
