;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns utilis.io
  (:require
   [clojure.java.io :as io]
   [utilis.timer :as timer])
  (:import
   [java.io
    ByteArrayInputStream
    File
    InputStream
    OutputStream]
   [java.nio ByteBuffer]
   [java.nio.channels Channels ReadableByteChannel WritableByteChannel]))

(defn file
  ^File
  [& args]
  (let [^File file (apply io/file args)]
    (-> file .getCanonicalPath io/file)))

(def resource io/resource)

(defn input-stream
  "Like clojure.java.io, but does not treat strings like URLs"
  ^InputStream
  [v]
  (if (string? v)
    (ByteArrayInputStream. (.getBytes ^String v))
    (io/input-stream v)))

(defn output-stream
  ^OutputStream
  [v]
  (io/output-stream v))

(defn copy
  [input output
   & {:keys [progress-fn progress-interval-ms buffer-size]
      :or {progress-interval-ms 1000
           buffer-size (* 128 1024)}}]
  (let [[input-opened ^InputStream input-stream]
        (if (instance? File input)
          [true (io/input-stream input)]
          [false input])
        [output-opened ^OutputStream output-stream]
        (if (instance? File output)
          [true (io/output-stream output)]
          [false output])
        buffer (ByteBuffer/allocateDirect buffer-size)
        bytes-copied (volatile! 0)
        progress-timer (when progress-fn
                         (timer/run-every
                          #(progress-fn @bytes-copied)
                          progress-interval-ms))]
    (try
      (with-open [^ReadableByteChannel input-channel (Channels/newChannel input-stream)
                  ^WritableByteChannel output-channel (Channels/newChannel output-stream)]
        (loop []
          (let [read-bytes (.read input-channel buffer)]
            (when (pos? read-bytes)
              (.flip buffer)
              (.write output-channel buffer)
              (.compact buffer)
              (vswap! bytes-copied + read-bytes)
              (recur))))
        ;; an eof can leave bytes in the buffer which must be drained
        (.flip buffer)
        (while (.hasRemaining buffer)
          (.write output-channel buffer)))
      (finally
        (some-> progress-timer timer/cancel)
        (when input-opened (.close input-stream))
        (when output-opened (.close output-stream))))))
