;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.netty.h2.client
  (:require [utilis.map :refer [map-vals]])
  (:import [java.net.http HttpClient HttpRequest HttpRequest$Builder HttpClient$Version HttpResponse$BodyHandlers]
           [java.util.concurrent Executors ExecutorService]
           [java.net URI]
           [java.io IOException]
           [java.net SocketException]))

(declare retry-request
         default-http-client
         default-http-client-executor)

(defn request
  [{:keys [url headers method client connection-reset-retries]
    :or {method :get
         connection-reset-retries 1}
    :as args}]
  (try (let [^HttpClient client (or client (or client @@default-http-client))
             ^HttpRequest$Builder builder (cond-> (HttpRequest/newBuilder)
                                            true (.uri (URI/create url))
                                            true (as-> %
                                                     (condp = method
                                                       :get (.GET %)
                                                       ;; :post (.POST %)
                                                       ;; :put (.PUT %)
                                                       ;; :options (.OPTIONS %)
                                                       )))
             builder (if (seq headers)
                       (.headers builder ^"[Ljava.lang.String;" (into-array (mapcat identity headers)))
                       builder)
             request (.build builder)
             response (.send client request (HttpResponse$BodyHandlers/ofByteArray))]
         {:status (.statusCode response)
          :body (.body response)
          :headers (-> (->> (.headers response)
                            .map
                            (into {})
                            (map-vals first))
                       (dissoc ":status"))})
       (catch IOException e
         (->> (dec connection-reset-retries)
              (assoc args :connection-reset-retries)
              (retry-request e)))
       (catch SocketException e
         (->> (dec connection-reset-retries)
              (assoc args :connection-reset-retries)
              (retry-request e)))))

(defn http-client
  ([] (http-client @default-http-client-executor))
  ([^ExecutorService executor]
   (-> (HttpClient/newBuilder)
       (.version HttpClient$Version/HTTP_2)
       (.executor executor)
       (.build))))

;;; Private

(def default-http-client-executor
  (delay
    (Executors/newFixedThreadPool
     (* 2 (.availableProcessors (Runtime/getRuntime))))))

(def default-http-client
  (atom (delay (http-client))))

(defn reset-default-client!
  []
  (reset! default-http-client (delay (http-client))))

(defn retry-request
  [^Throwable e {:keys [connection-reset-retries] :as args}]
  (if (pos? connection-reset-retries)
    (do (reset-default-client!)
        (request args))
    (throw e)))
