;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.netty.h2.handlers.settings
  (:import [io.netty.channel ChannelHandlerContext]
           [io.netty.handler.codec.http2
            Http2CodecUtil
            Http2SettingsFrame
            Http2SettingsAckFrame
            Http2Settings]))

(def settings-map
  {(char 1) :settings-header-table-size
   (char 2) :settings-enable-push
   (char 3) :settings-max-concurrent-streams
   (char 4) :settings-initial-window-size
   (char 5) :settings-max-frame-size
   (char 6) :settings-max-header-list-size})

(def settings-map-keys (keys settings-map))

(defn settings
  [^Http2SettingsFrame frame]
  (let [^Http2Settings settings (.settings frame)]
    (->> settings-map-keys
         (reduce (fn [m k]
                   (if-let [v (.get settings k)]
                     (assoc! m (get settings-map k) v)
                     m))
                 (transient {}))
         persistent!)))

(defn handle?
  [msg]
  (or (instance? Http2SettingsFrame msg)
      (instance? Http2SettingsAckFrame msg)))

(defn handle
  [^ChannelHandlerContext _ctx connection-state msg]
  (cond
    (instance? Http2SettingsFrame msg)
    (let [{:keys [settings-initial-window-size]
           :as settings} (merge {:settings-initial-window-size Http2CodecUtil/DEFAULT_WINDOW_SIZE
                                 :settings-max-frame-size (or (get @connection-state [:server-args :default-outbound-max-frame-size])
                                                              Http2CodecUtil/DEFAULT_MAX_FRAME_SIZE)}
                                (settings msg))]

      (swap! connection-state assoc
             :settings settings
             :window (volatile! settings-initial-window-size)))
    (instance? Http2SettingsAckFrame msg)
    (swap! connection-state assoc :settings-ack true))
  nil)
