;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.netty.http1.websocket
  (:refer-clojure :exclude [send])
  (:require [vectio.netty :as n]
            [vectio.netty.websocket :as nws]
            [manifold.stream :as s]
            [manifold.deferred :as d])
  (:import [io.netty.handler.codec.http2 Http2SecurityUtil]
           [io.netty.bootstrap Bootstrap]
           [io.netty.channel.socket SocketChannel]
           [io.netty.channel
            SimpleChannelInboundHandler
            ChannelInitializer
            ChannelDuplexHandler
            ChannelHandler
            ChannelHandlerContext
            ChannelPromise]
           [io.netty.handler.codec.http
            FullHttpResponse
            DefaultHttpHeaders
            HttpClientCodec
            HttpObjectAggregator]
           [io.netty.channel.nio NioEventLoopGroup]
           [io.netty.channel.socket.nio NioSocketChannel]
           [io.netty.handler.codec.http.websocketx
            WebSocketHandshakeException
            WebSocketClientHandshakerFactory
            WebSocketClientHandshaker
            WebSocketVersion]
           [io.netty.handler.ssl
            SslContextBuilder
            SslContext
            SslProvider
            SupportedCipherSuiteFilter
            ApplicationProtocolConfig
            ApplicationProtocolConfig$Protocol
            ApplicationProtocolConfig$SelectorFailureBehavior
            ApplicationProtocolConfig$SelectedListenerFailureBehavior
            ApplicationProtocolNames]
           [io.netty.handler.ssl.util
            InsecureTrustManagerFactory]
           [java.net URI]))

(declare websocket-client-handler channel-initializer)

(defn websocket-client
  [{:keys [host port ssl-context path
           on-text-message
           on-binary-message
           on-ping-message
           on-pong-message
           on-close]}]
  (let [deferred (d/deferred)
        worker-group (NioEventLoopGroup.)]
    (try
      (let [handshake-future (atom nil)
            ^ChannelHandler handler (websocket-client-handler
                                     (WebSocketClientHandshakerFactory/newHandshaker
                                      (URI. (str (if ssl-context
                                                   "wss"
                                                   "ws")
                                                 "://"
                                                 host
                                                 ":"
                                                 port
                                                 (when (not (re-find #"^/" path))
                                                   "/")
                                                 path))
                                      WebSocketVersion/V13
                                      nil
                                      false
                                      (DefaultHttpHeaders.))
                                     handshake-future
                                     {:on-text-message on-text-message
                                      :on-binary-message on-binary-message
                                      :on-ping-message on-ping-message
                                      :on-pong-message on-pong-message
                                      :on-close on-close})
            b (doto (Bootstrap.)
                (.group worker-group)
                (.channel ^NioSocketChannel NioSocketChannel)
                (.handler (channel-initializer host port ssl-context handler)))
            channel (-> b
                        (.connect (str host) (int port))
                        .sync
                        .channel)
            max-flush-size (dec (int (Math/pow 2 16)))
            max-frame-size (dec (int (Math/pow 2 16)))]
        (.sync ^ChannelPromise @handshake-future)
        (d/success! deferred
                    {:close (fn [] (.shutdownGracefully worker-group))
                     :send #(n/safe-execute
                             channel (fn []
                                       (->> %
                                            (nws/data->websocket-frames channel max-frame-size)
                                            (nws/send-websocket-frames channel max-flush-size))))}))
      (catch Exception e
        (.shutdownGracefully worker-group)
        (d/error! deferred e)))
    deferred))

(defn websocket-client-stream
  [{:keys [host port ssl-context path] :as args}]
  (let [out (s/stream)
        in (s/stream)
        spliced (s/splice out in)
        deferred (d/deferred)]
    (d/on-realized
     (websocket-client
      (assoc args
             :on-close #(s/close! spliced)
             :on-text-message #(do @(s/put! in %))
             :on-binary-message #(do @(s/put! in %))
             :on-ping-message (fn [_] @(s/put! in :ping))
             :on-pong-message (fn [_] @(s/put! in :pong))))
     (fn [{:keys [send close]}]
       (s/consume send out)
       (s/on-closed out (fn [] (close)))
       (d/success! deferred spliced))
     (fn [error]
       (s/close! spliced)
       (s/close! in)
       (d/error! deferred error)))
    deferred))

(defn send
  [{:keys [send]} text-message]
  (send text-message))

(defn close
  [{:keys [close]}]
  (close))

(defn unsafe-self-signed-ssl-context
  []
  (-> (SslContextBuilder/forClient)
      (.sslProvider SslProvider/JDK)
      (.ciphers Http2SecurityUtil/CIPHERS
                SupportedCipherSuiteFilter/INSTANCE)
      (.trustManager InsecureTrustManagerFactory/INSTANCE)
      (.applicationProtocolConfig (ApplicationProtocolConfig.
                                   ApplicationProtocolConfig$Protocol/ALPN
                                   ApplicationProtocolConfig$SelectorFailureBehavior/NO_ADVERTISE
                                   ApplicationProtocolConfig$SelectedListenerFailureBehavior/ACCEPT
                                   ^"[Ljava.lang.String;" (into-array [ApplicationProtocolNames/HTTP_1_1])))
      (.build)))

;;; Private

(defn websocket-client-handler
  ^ChannelDuplexHandler
  [^WebSocketClientHandshaker handshaker
   handshake-future
   {:keys [on-text-message
           on-binary-message
           on-ping-message
           on-pong-message
           on-close]}]
  (let [frame-handler (nws/inbound-collector
                       {:on-text-message on-text-message
                        :on-binary-message on-binary-message
                        :on-ping-message on-ping-message
                        :on-pong-message on-pong-message
                        :on-close on-close})]
    (proxy [SimpleChannelInboundHandler] []
      (handlerAdded [^ChannelHandlerContext ctx]
        (reset! handshake-future (.newPromise ctx)))
      (channelActive [^ChannelHandlerContext ctx]
        (.handshake handshaker (.channel ctx)))
      (channelInactive [^ChannelHandlerContext ctx]
        ;; client disconnected
        (when on-close
          (on-close)))
      (channelRead0 [^ChannelHandlerContext ctx msg]
        (let [ch (.channel ctx)]
          (cond
            (not (.isHandshakeComplete handshaker))
            (try
              (.finishHandshake handshaker ch ^FullHttpResponse msg)
              (.setSuccess ^ChannelPromise @handshake-future)
              (catch WebSocketHandshakeException e
                (.setFailure ^ChannelPromise @handshake-future e)))

            (instance? FullHttpResponse msg)
            (throw (ex-info "Unexpected FullHttpResponse" {:msg msg}))

            :else (when-let [handler (frame-handler ctx msg)]
                    (handler)))))
      (exceptionCaught [^ChannelHandlerContext ctx ^Throwable cause]
        (when (not (.isDone ^ChannelPromise @handshake-future))
          (.setFailure ^ChannelPromise @handshake-future cause))
        (.close ctx)))))

(defn channel-initializer
  ^ChannelInitializer [^String host ^long port ^SslContext ssl-context ^ChannelHandler handler]
  (proxy [ChannelInitializer] []
    (initChannel [^SocketChannel ch]
      (let [p (.pipeline ch)]
        (when ssl-context
          (.addLast p "ssl-handler" (.newHandler ssl-context (.alloc ch) host (int port))))
        (.addLast p "client-codec" (HttpClientCodec.))
        (.addLast p "http-object-aggregator" (HttpObjectAggregator. 8192))
        (.addLast p "client-handler" handler)))))
