(ns vectio.util.stream
  (:require #?(:clj [manifold.stream :as s]
               :cljs [vectio.util.manifold.stream :as s])
            #?(:clj [manifold.deferred :as d]
               :cljs [vectio.util.manifold.deferred :as d])
            [utilis.timer :as timer]
            #?(:cljs [utilis.js :as j])))

(defn log-error
  [& args]
  #?(:clj (locking Object
            (apply println args))
     :cljs (apply js/console.warn (clj->js args))))

(defn reconnecting-stream
  ([init-fn]
   (reconnecting-stream
    init-fn {:reconnect-timeout
             (fn [attempt]
               (min 5000 (* attempt 100)))}))
  ([init-fn {:keys [reconnect-timeout]}]
   (let [event-stream (s/stream)
         out (s/stream)
         in (s/stream)
         spliced (s/splice out in)
         underlying-stream-atom (atom nil)
         reconnect? (atom true)
         initial (atom true)
         cleanup (fn []
                   (reset! reconnect? false)
                   (s/close! event-stream)
                   (when-let [stream @underlying-stream-atom]
                     (when (not (s/closed? stream))
                       (s/close! stream))))
         init (fn init
                ([] (init 0))
                ([attempt]
                 (d/on-realized
                  (init-fn)
                  (fn [underlying-stream]
                    (reset! underlying-stream-atom underlying-stream)
                    (s/put! event-stream
                            {:underlying-stream underlying-stream
                             :stream spliced
                             :initial @initial
                             :state :connected})
                    (s/consume (fn [message] (s/put! in message)) underlying-stream)
                    (s/on-closed underlying-stream
                                 (fn []
                                   (reset! underlying-stream-atom nil)
                                   (s/put! event-stream {:state :disconnected
                                                         :error :underlying-stream-closed})
                                   (when @reconnect? (init))))
                    (when @initial (reset! initial false)))
                  (fn [error]
                    (s/put! event-stream {:error error
                                          :state :disconnected})
                    (when @reconnect?
                      (timer/run-after
                       #(init (inc attempt))
                       (cond
                         (number? reconnect-timeout)
                         reconnect-timeout

                         (fn? reconnect-timeout)
                         (reconnect-timeout attempt)

                         :else (throw (ex-info "reconnect-timeout must be a number or a function"
                                               {:reconnect-timeout reconnect-timeout})))))))))]
     (s/consume #(s/put! @underlying-stream-atom %) out)
     (s/on-closed out (fn [] (cleanup)))
     (init)
     event-stream)))
