(ns vectio.tcp
  (:require [fluxus.deferred :as d]
            [fluxus.stream :as s]
            [vectio.netty :as n])
  (:import [io.netty.bootstrap Bootstrap]
           [io.netty.buffer ByteBuf]
           [io.netty.channel
            ChannelHandlerContext
            ChannelInitializer
            SimpleChannelInboundHandler]
           [io.netty.channel.nio NioEventLoopGroup]
           [io.netty.channel.socket SocketChannel]
           [io.netty.channel.socket.nio NioSocketChannel]
           [java.net InetSocketAddress]))

(declare channel-inbound-handler)

(defn client
  [{:keys [host port]}]
  (let [group (NioEventLoopGroup.)
        out (s/stream)
        in (s/stream)
        paired-stream (s/paired-stream out in)
        deferred (d/deferred)]
    (s/on-close out (fn [_]
                      (.sync (.shutdownGracefully group))
                      (s/close! in)
                      (s/close! paired-stream)))
    (try
      (let [client-bootstrap (Bootstrap.)]
        (doto client-bootstrap
          (.group group)
          (.channel ^Class NioSocketChannel)
          (.remoteAddress (InetSocketAddress. ^String host ^long port))
          (.handler (proxy [ChannelInitializer] []
                      (initChannel [^SocketChannel socket-channel]
                        (.addLast (.pipeline socket-channel) "inbound-handler"
                                  ^SimpleChannelInboundHandler (channel-inbound-handler in out))))))
        (.sync (.connect client-bootstrap))
        (d/success! deferred paired-stream))
      (catch Exception e
        (d/error! deferred e)
        (s/close! out)))
    deferred))


;;; Private

(defn- channel-inbound-handler
  ^SimpleChannelInboundHandler [in out]
  (proxy [SimpleChannelInboundHandler] []
    (channelActive [^ChannelHandlerContext ctx]
      (s/consume
       #(let [byte-buf (n/to-byte-buf ctx)]
          (n/safe-execute
           ctx (fn []
                 (->> %
                      byte-buf
                      (.writeAndFlush ctx)))))
       out))
    (channelRead0 [^ChannelHandlerContext ctx msg]
      (let [content ^ByteBuf msg
            bytes (byte-array (.readableBytes content))]
        (.readBytes content bytes)
        (s/put! in bytes)))))
