;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.http
  (:refer-clojure :exclude [get])
  (:require [vectio.netty.server :as server]
            [vectio.netty.http1.websocket :as http1-websocket]
            [hato.client :as hc]
            [clojure.set :refer [rename-keys]]
            [utilis.fn :refer [fsafe]]
            [clojure.java.io :as io]
            [integrant.core :as ig])
  (:import [java.net InetSocketAddress]
           [javax.net.ssl SSLParameters]
           [io.netty.handler.ssl
            ApplicationProtocolConfig
            ApplicationProtocolConfig$Protocol
            ApplicationProtocolConfig$SelectorFailureBehavior
            ApplicationProtocolConfig$SelectedListenerFailureBehavior
            ApplicationProtocolNames
            SslContextBuilder
            SslProvider
            ClientAuth
            SupportedCipherSuiteFilter
            JdkSslContext]
           [io.netty.handler.codec.http2 Http2SecurityUtil]
           [java.net InetSocketAddress]
           [java.io InputStream ByteArrayInputStream File Closeable]))

(declare server ensure-http-client)

(defmethod ig/init-key :vectio.http/server
  [_ opts]
  (let [opts (rename-keys opts {:port :http-port
                                :host :http-host})
        ring-handler (atom (delay (:ring-handler opts)))
        {:keys [http-host http-port tls] :or {http-host "localhost"}} opts]
    (when (not http-port)
      (throw (ex-info "An http-port must be provided to start :vectio.http/server"
                      {:http-host http-host
                       :http-port http-port})))
    {:ring-handler ring-handler
     :http-server (server
                   (merge
                    (select-keys opts [:leak-detector-level
                                       :initial-window-size
                                       :websocket-max-frame-size
                                       :max-frame-size
                                       :max-flush-size
                                       :max-concurrent-streams
                                       :max-header-list-size
                                       :push-enabled
                                       :default-outbound-max-frame-size
                                       :protocols])
                    {:handler (fn [request] ((fsafe @@ring-handler) request))
                     :socket-address (InetSocketAddress. ^String http-host ^int http-port)
                     :ssl-context (when (seq tls)
                                    (let [coerce-is (fn [v]
                                                      (cond
                                                        (instance? InputStream v) v
                                                        (string? v) (ByteArrayInputStream. (.getBytes ^String v))
                                                        (instance? File v) (io/input-stream v)))]
                                      (-> (cond-> (SslContextBuilder/forServer
                                                   ^InputStream (coerce-is (get-in tls [:server :cert]))
                                                   ^InputStream (coerce-is (get-in tls [:server :key])))
                                            (get-in tls [:client :ca])
                                            (-> (.clientAuth (ClientAuth/REQUIRE))
                                                (.trustManager ^InputStream (coerce-is (get-in tls [:client :ca])))))
                                          (.sslProvider SslProvider/JDK)
                                          (.ciphers Http2SecurityUtil/CIPHERS SupportedCipherSuiteFilter/INSTANCE)
                                          (.applicationProtocolConfig
                                           (ApplicationProtocolConfig.
                                            ApplicationProtocolConfig$Protocol/ALPN
                                            ApplicationProtocolConfig$SelectorFailureBehavior/NO_ADVERTISE
                                            ApplicationProtocolConfig$SelectedListenerFailureBehavior/ACCEPT
                                            ^"[Ljava.lang.String;" (into-array [ApplicationProtocolNames/HTTP_2])))
                                          .build)))}))}))

(defmethod ig/halt-key! :vectio.http/server
  [_ {:keys [http-server]}]
  (when http-server
    (.close ^Closeable http-server)))

(defmethod ig/suspend-key! :vectio.http/server
  [_ {:keys [ring-handler]}]
  (reset! ring-handler (promise)))

(defmethod ig/resume-key :vectio.http/server
  [key opts old-opts old-impl]
  (if (= (dissoc opts :ring-handler) (dissoc old-opts :ring-handler))
    (do (deliver @(:ring-handler old-impl) (:ring-handler opts))
        old-impl)
    (do (ig/halt-key! key old-impl)
        (ig/init-key key opts))))

(defn server
  [{:keys [ssl-context
           socket-address
           host port
           handler
           leak-detector-level]
    :as server-args}]
  (server/start-server
   (cond-> server-args
     (and (not socket-address) host port)
     (update :socket-address
             #(or % (InetSocketAddress. ^String host (int port)))))))

(defn websocket-client
  [{:keys [host port tls path]
    :as websocket-args}]
  (http1-websocket/websocket-client-stream websocket-args))

(defn websocket-stream-response
  [request]
  (server/websocket-stream-response request))

(defn client
  [{:keys [tls ssl-context]}]
  (hc/build-http-client
   (merge
    {:connect-timeout 10000}
    (when (or tls ssl-context)
      {:ssl-parameters
       (doto (SSLParameters.)
         (.setNeedClientAuth true))
       :ssl-context
       (or ssl-context
           (when (seq tls)
             (let [coerce-is
                   (fn [v]
                     (cond
                       (instance? InputStream v) v
                       (string? v) (ByteArrayInputStream.
                                    (.getBytes ^String v))
                       (instance? File v) (io/input-stream v)))
                   ^JdkSslContext ssl-context
                   (-> (SslContextBuilder/forClient)
                       (.trustManager ^InputStream (coerce-is (get-in tls [:server :ca])))
                       (.keyManager ^InputStream (coerce-is (get-in tls [:client :cert]))
                                    ^InputStream (coerce-is (get-in tls [:client :key])))
                       (.build))]
               (.context ssl-context))))}))))

(defn get
  ([client url] (get client url nil))
  ([client url options]
   (hc/get url (assoc options :http-client client))))

(defn post
  ([client url] (post client url nil))
  ([client url options]
   (hc/post url (assoc options :http-client client))))

(defn put
  ([client url] (put client url nil))
  ([client url options]
   (hc/put url (assoc options :http-client client))))

(defn patch
  ([client url] (patch client url nil))
  ([client url options]
   (hc/patch url (assoc options :http-client client))))

(defn delete
  ([client url] (delete client url nil))
  ([client url options]
   (hc/delete url (assoc options :http-client client))))

(defn head
  ([client url] (head client url nil))
  ([client url options]
   (hc/head url (assoc options :http-client client))))

(defn options
  ([client url] (options client url nil))
  ([client url options]
   (hc/options url (assoc options :http-client client))))
