;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.util.gzip
  (:require [utilis.map :refer [map-keys]]
            [clojure.string :as st])
  (:import [java.util.zip GZIPOutputStream]
           [java.io ByteArrayOutputStream File InputStream FileInputStream]
           [org.apache.commons.io IOUtils]))

(defn test-array
  [t]
  (let [check (type (t []))]
    (fn [arg] (instance? check arg))))

(def byte-array?
  (test-array byte-array))

(defn to-byte-array
  [x]
  (cond
    (byte-array? x) x

    (instance? String x)
    (IOUtils/toByteArray x)

    (instance? File x)
    (with-open [fis (FileInputStream. ^File x)]
      (IOUtils/toByteArray fis))

    (instance? InputStream x)
    (IOUtils/toByteArray ^InputStream x)

    :else nil))

(defn gzip
  [^bytes input]
  (let [baos (ByteArrayOutputStream.)
        gzos (GZIPOutputStream. baos)]
    (.write gzos input)
    (.close gzos)
    (.close baos)
    (.toByteArray baos)))

(defn lower-case-headers
  [response]
  (update response :headers
          (partial map-keys
             (comp st/lower-case
                #(if (keyword? %)
                   (name %)
                   (str %))))))

(defn gzip-handler
  [handler]
  (fn [request]
    (let [response (handler request)]
      (or (let [response (lower-case-headers response)]
            (when (and (:body response)
                       (re-find #"gzip" (str (get-in request [:headers "accept-encoding"])))
                       (not (get-in response [:headers "content-encoding"])))
              (when-let [body (to-byte-array (:body response))]
                (let [body (gzip body)]
                  (-> response
                      (assoc :body body)
                      (assoc-in [:headers "content-encoding"] "gzip")
                      (assoc-in [:headers "content-length"] (count body)))))))
          response))))
