;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.haslett.websocket
  (:refer-clojure :exclude [subs])
  (:require [fluxus.flow :as f]
            [fluxus.promise :as p]
            [cljs.core.async :as async]
            [haslett.client :as ws]
            [clojure.string :as st]))

(defn websocket-client
  [{:keys [address on-text-message on-close]}]
  (let [client (p/promise)]
    (async/go
      (try
        (let [return (ws/connect address)
              stream (async/<! return)]
          (if (ws/connected? stream)
            (let [{:keys [close-status source]} stream]
              (async/go
                (try (loop []
                       (async/alt!
                         source ([message]
                                 (do (on-text-message message)
                                     (recur)))
                         close-status ([status]
                                       (on-close status))))
                     (catch js/Error e
                       (js/console.error "Error occurred in vectio.haslett.websocket/websocket-client read loop" e))))
              (p/resolve! client
                          {:stream stream
                           :close (fn [] (ws/close stream))
                           :send (fn [^js/String message]
                                   (async/put! (:sink stream) message))}))
            (let [close-status (async/<! (:close-status stream))
                  reason (st/lower-case (str (:reason close-status)))]
              (p/reject! client
                         (assoc close-status :fatal?
                                (boolean
                                 (or (re-find #"java.security.cert.CertPathValidatorException" reason)
                                     (re-find #"expected http 101 response" reason)
                                     (re-find #"403" reason))))))))
        (catch js/Error e
          (js/console.error "Error occurred in vectio.haslett.websocket/websocket-client" e))))
    client))

(defn websocket-client-stream
  [{:keys [address] :as args}]
  (let [[client-stream internal] (f/entangled)
        message-stream (p/promise)
        client (websocket-client
                (assoc args
                       :on-close #(f/close! internal)
                       :on-text-message #(f/put! internal %)))]
    (-> client
        (p/then (fn [{:keys [send close]}]
                  (f/consume send internal)
                  (f/on-close client-stream (fn [_] (close)))
                  (p/resolve! message-stream client-stream)))
        (p/catch (fn [error]
                   (f/close! internal)
                   (p/reject! message-stream error))))
    message-stream))
