;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.middleware.gzip
  (:require [ring.util.io :as r-io]
            [utilis.io :as io])
  (:import [java.util.zip GZIPOutputStream]
           [java.io InputStream ByteArrayOutputStream
            File]))

(declare compress)

(defn middleware
  [handler]
  (fn [request]
    (let [response (handler request)]
      (cond-> response
        (and (:body response)
             (not (get-in response [:headers "content-encoding"]))
             (when-let [encoding (get-in request [:headers "accept-encoding"])]
               (re-find #"gzip" encoding)))
        compress))))

(defn- compress
  [{:keys [body] :as response}]
  (let [byte-array? (fn [b]
                      (instance? (type (byte-array 0)) b))
        compress-bytes #(let [body (cond-> body (string? body) .getBytes)
                              bytes (with-open [ba-os (ByteArrayOutputStream.)
                                                gz-os (GZIPOutputStream. ba-os)]
                                      (.write gz-os body)
                                      (.toByteArray ba-os))]
                          (-> response
                              (assoc :body bytes)
                              (update :headers (fn [headers]
                                                 (-> headers
                                                     (assoc "content-encoding" "gzip"
                                                            "content-length" (count bytes)))))))
        compress-stream #(-> response
                             (assoc :body (r-io/piped-input-stream
                                           (fn [os]
                                             (with-open [gzos (GZIPOutputStream. os)]
                                               (io/copy body gzos)))))
                             (update :headers (fn [headers]
                                                (-> headers
                                                    (assoc "content-encoding" "gzip")
                                                    (dissoc "content-length")))))]
    (cond
      (or (byte-array? body) (string? body))
      (compress-bytes)

      (or (instance? InputStream body) (instance? File body))
      (compress-stream))))
